
param(
  [string]$PingTarget = "www.addplm.com"
)


# Result: 5 metrics, each containing 20 averaged samples. Job duration: 10 minutes
# $measuresCount = $Env:measuresCount
# How many PingMeasument are done and send to PLMPerfService
$measuresCount = 1
# Duration for to execute one PingMeasument in this case = 60 Sec
$averagingDurationSeconds = 60
# how many pings request to host are send during $averagingDurationSeconds
# Exa: 60 sec/20 Pings  3 sec/Ping every 3 second one ping is done
$pingCountPerAveragingMeasurement = 20

$measuresCount = 1
$averagingDurationSeconds = 10
$pingCountPerAveragingMeasurement = 5

if ($env:JobMgr_Debug -eq "ON") {
    $measuresCount = 1
    $averagingDurationSeconds = 10
    $pingCountPerAveragingMeasurement = 5
    Write-Output "!! Debug-Mode active, returning only 2 measurements of each 5 seconds !!"
}

$payloadSizeBytes = 32

$totalSeconds = $measuresCount * $averagingDurationSeconds
$totalTime = New-TimeSpan -Seconds $totalSeconds
$endTimestamp = (Get-Date).Add($totalTime)

Write-Output ""
Write-Output "----------------------------------------------------------------------------------------"
Write-Output "Starting PingJob to $PingTarget"
Write-Output "----------------------------------------------------------------------------------------"

Write-Output ""
Write-Output "Executing $measuresCount measurements, each of them with $pingCountPerAveragingMeasurement ping requests."
Write-Output "This job will take in total $([Math]::Floor($totalTime.TotalMinutes)) minutes and $($totalTime.Seconds) seconds until $('{0:HH:mm:ss}' -f (Get-Date).Add($totalTime))."
Write-Output ""

######################

# determine ping target ip address
$dns = Resolve-DnsName $PingTarget
$aRecord = $dns | Where { $_.Type -eq "A" } | Select -First 1
$aaaaRecord = $dns | Where { $_.Type -eq "AAAA" } | Select -First 1
# prefer IPv4 over IPv6, fallback to PingTarget (may be IP already given as IP-address)
if     ($aRecord)    { $TargetIpAddress = $aRecord.IPAddress}
elseif ($aaaaRecord) { $TargetIpAddress = $aaaaRecord.IPAddress}
else                 { $TargetIpAddress = $PingTarget }

# determine source ip address
$routeToTarget = Find-NetRoute -RemoteIPAddress $TargetIpAddress | Select -First 1
$sourceInterface = $routeToTarget.InterfaceAlias
$sourceIpAddress = $routeToTarget.IPAddress

# set route information header
$route = "Source: $sourceIpAddress (Interface $sourceInterface)`r`nTarget: $TargetIpAddress ($PingTarget)"


# Load MeasureJobResults.dll
$assemblyNE = "JobManagerContract.MeasureJobResults.dll"
$assemblyPath1 = Join-Path $PSScriptRoot $assemblyNE
$assemblyPath2 = Join-Path $Env:JobClient_ServerProvidedResources_DP $assemblyNE
Write-Output "AssemblyPath 1: $assemblyPath1"
Write-Output "AssemblyPath 2: $assemblyPath2"

$assemblyPath = $assemblyPath1
if (-not (Test-Path $assemblyPath)) { $assemblyPath = $assemblyPath2 }
if (-not $assemblyPath -or -not (Test-Path $assemblyPath)) {
   Write-Error "Assembly JobManagerContract.MeasureJobResults.dll not found."
   if ($psISE) { Return }
   Exit 1
}

Write-Output "Loading assembly: $assemblyPath"
[Reflection.Assembly]::LoadFile($assemblyPath) | Out-Null

# Report header
$measureJobResults = New-Object -TypeName "JF.PlmJobManager.JobServer.MeasureJobResults_cls"
$measureJobResults.AdditionalData.AddString("SourceInterface", $sourceInterface)
$measureJobResults.AdditionalData.AddString("SourceIpAddress", $sourceIpAddress)
$measureJobResults.AdditionalData.AddString("PingTarget", $PingTarget)
$measureJobResults.AdditionalData.AddString("TargetIpAddress", $TargetIpAddress)

# initiate background job to evaluate the route to the host (only if PS version supports the cmdlet)
$hopCount = -1
$tracerouteJob = $null
if (Get-Command Test-NetConnection -ErrorAction SilentlyContinue) {
    $tracerouteJob = Start-Job -ScriptBlock {
        param ($traceToHost)
        $OutputEncoding = [System.Text.Encoding]::UTF8
        $tracert = & tracert -w 2000 $traceToHost | Out-String
        return $tracert
        #$ProgressPreference = 'SilentlyContinue'
        #$route = Test-NetConnection -ComputerName $traceToHost -TraceRoute
        #return $route
    } -ArgumentList $TargetIpAddress
}

$measureNumber = 0
$nextPingStart = [System.DateTime]::MinValue
$durationPerPingMilliseconds = 1000.0 * $averagingDurationSeconds / $pingCountPerAveragingMeasurement
[System.Collections.ArrayList]$measureResults = @()

Write-Output "Measuring $measuresCount average ping results to $PingTarget using address $TargetIpAddress."
Write-Output "Each measurement will aggregate $pingCountPerAveragingMeasurement single ping results within $averagingDurationSeconds seconds."
Write-Output "The total duration will be $($measuresCount * $averagingDurationSeconds) seconds."
Write-Output "---"

while($measureNumber -lt $measuresCount) {
  $measureNumber = $measureNumber + 1

  $pingNumber = 0
  [System.Collections.ArrayList]$pingLatencies = @()
  $pingsSent = 0
  $pingsReceived = 0
  $pingsLost = 0

  while ($pingNumber -lt $pingCountPerAveragingMeasurement) {
    $pingNumber = $pingNumber + 1

    # wait until next ping
    if ((Get-Date) -lt $nextPingStart) {
        $waitTimeSpan = $nextPingStart - (Get-Date)
        Write-Output "Waiting for next ping $pingNumber in group $measureNumber :: $waitTimeSpan"
        Start-Sleep -Milliseconds $waitTimeSpan.TotalMilliseconds
    }
    
    $nextPingStart = (Get-Date).AddMilliseconds($durationPerPingMilliseconds)
    try {
        $pingsSent += 1
        $pingResult = Test-Connection -ComputerName $TargetIpAddress -BufferSize $payloadSizeBytes -Count 1 -ErrorAction Stop
        if ($pingResult) {
            $pingsReceived += 1
            $pingLatencies.Add($pingResult.ResponseTime) | Out-Null
        } else {
            $pingsLost += 1
        }
    } catch {
        $pingsLost += 1
    }

  }

  
  # Calculate Avg, Min, Max, Count
  $measuredLatencies = $pingLatencies | Measure -Average -Minimum -Maximum

  [double]$stdevTemp = $null
  [double]$stdev = 0
  
  # Calculate StDev only if two or more results are available
  if ($pingLatencies.Count -ge 2)
  {
	  # Iterate through each of the numbers and get part of the variance via some PowerShell math.
	  ForEach($pingLatency in $pingLatencies) {
		$stdevTemp += [Math]::Pow(($pingLatency - $measuredLatencies.Average), 2)
	  }
	  #Finish the variance calculation, and get the square root to finally get the standard deviation.
	  $stdev = [math]::Sqrt($($stdevTemp / ($measuredLatencies.Count - 1)))
  }

  # Query traceroute job results
  if ($tracerouteJob) {
    
    # Wait for job completion
    Wait-Job -Id $tracerouteJob.Id -Timeout 60 | Out-Null
    $job = Get-Job -Id $tracerouteJob.Id

    # Only get the job if it has completed, otherwise ignore
    if ($job.State -eq "Completed") {
        # Retrieve the results
        $tracertOutput = Receive-Job -Id $tracerouteJob.Id
        #$traceRoute = $JobResults.TraceRoute
        #$hopCount = $traceRoute.Count
        #$route = ConvertTo-Json $traceRoute

        # SAMPLE OUTPUT:
        #  Routenverfolgung zu www.addplm.com [217.160.0.225] ber maximal 30 Hops:
        # 
        #   1    <1 ms    <1 ms    <1 ms  fritzbox.homenet.local [192.168.0.254]
        #   2    10 ms    10 ms    10 ms  p3e9bf664.dip0.t-ipconnect.de [62.155.246.100]
        #   3    14 ms    13 ms    13 ms  f-ed12-i.F.DE.NET.DTAG.DE [62.154.17.158]
        #   4    13 ms    13 ms    13 ms  80.157.128.146
        #   5     *        *        *     TIMEOUT
        #   6    19 ms    18 ms    18 ms  lo-0.gw-distd-sh-1.bap.rhr.de.net.ionos.com [212.227.112.225]
        #   7    19 ms    18 ms    18 ms  217-160-0-225.elastic-ssl.ui-r.com [217.160.0.225]
        # 
        # Ablaufverfolgung beendet.

        # SAMPLE PROCESSED OUTPUT:
        # Routenverfolgung zu www.addplm.com [217.160.0.225] ber maximal 30 Hops:
        #
        #   1 :: 192.168.0.254   fritzbox.homenet.local
        #   2 :: 62.155.246.100  p3e9bf664.dip0.t-ipconnect.de
        #   3 :: 62.154.17.158   f-ed12-i.F.DE.NET.DTAG.DE
        #   4 :: 80.157.128.146
        #   5 :: *** Timeout waiting for response ***
        #   6 :: 212.227.112.225 lo-0.gw-distd-sh-1.bap.rhr.de.net.ionos.com
        #   7 :: 217.160.0.225   217-160-0-225.elastic-ssl.ui-r.com
        #   
        # Ablaufverfolgung beendet.

        # Process the output line by line and apply custom padding for IP
        $tracertOutput = $tracertOutput -replace "`r`n", "`n"
        $formattedOutput = $tracertOutput -split "`n" | ForEach-Object {
            
            # fetch highest hop number as hop count
            if ($_ -match "^\s+(?<hop>\d+)") {
                $hopNum = [int]$matches['hop']
                if ($hopNum -gt $hopCount) { $hopCount = $hopNum }
            }

            if ($_ -match "^\s*(?<hop>\d+)(\s+(\<?\s*\d+ ms|\*)){3}\s+((?<host>[^\s]+)\s+[\[\(](?<ip>\d+\.\d+\.\d+\.\d+)[\]\)]|(?<ip>\d+\.\d+\.\d+\.\d+)).*$") {
                # Extract matches
                $hop = $matches['hop']
                $ip = $matches['ip']
                $hostname = $matches['host']

                # Pad IP to always be 15 characters long
                $hop = $hop.PadLeft(3, ' ')
                $ip = $ip.PadRight(15, ' ')

                # Format the replacement string
                return "$hop :: $ip $hostname".TrimEnd()

            } elseif ($_ -match "^\s*(?<hop>\d+)(\s+\*){3}\s+.*$") {
                # Extract matches
                $hop = $matches['hop']
                $hop = $hop.PadLeft(3, ' ')
                "$hop :: *** Timeout waiting for response ***"
            
            } else {
                $_  # Keep non-matching lines as-is
            
            }
        }

        $route = "$($route)`r`n$($formattedOutput -join "`r`n")"
    }

    # Stop if running
    if ($job.State -eq "Running") {
        Stop-Job -Id $tracerouteJob.Id | Out-Null
    }

    # Cleanup
    Remove-Job -Id $tracerouteJob.Id
    $tracerouteJob = $null
  }

  $measureResult = $measureJobResults.AddNewMeasureResult([System.DateTime]::UtcNow, $measuredLatencies.Average)
  $measureResult.AdditionalData.AddInteger("SentCount", $pingsSent)
  $measureResult.AdditionalData.AddInteger("ReceivedCount", $pingsReceived)
  $measureResult.AdditionalData.AddInteger("MinLatency", [int]($measuredLatencies.Minimum))
  $measureResult.AdditionalData.AddInteger("MaxLatency", [int]($measuredLatencies.Maximum))
  $measureResult.AdditionalData.AddDouble("AverageLatency", $measuredLatencies.Average)
  $measureResult.AdditionalData.AddDouble("AverageLatencyStdev", $stdev)
  
}

$measureJobResults.JobResultCode = 0
$measureJobResults.JobResultMessage = "OK"
$measureJobResults.JobResultReport = ""
  
$measureJobResults.AdditionalData.AddInteger("PayloadSizeBytes", $payloadSizeBytes)
$measureJobResults.AdditionalData.AddInteger("HopCount", $hopCount)
$measureJobResults.AdditionalData.AddString("Route", $route)

if ($env:JobMgr_Debug -eq "ON") {
  Write-Output "Route to target:"
  Write-Output $route
}

#Write-Output "Saving to $($PSScriptRoot)"
$xml = $measureJobResults.SaveAsXmlFile($PSScriptRoot)
#$xml | Out-Host


