###############################################################################
# phys_material.tcl - DBC Event Handler for material DB stored as ascii file
###############################################################################
#
# There are three material record types in the material library. The first
# five fields are the same in each:
#
#         Libref Name Category Type Ref_Temp
#
# The Version field has been added since. This is to be found as the last field
#         of each record.
#
# When Type == ISO (ISOTROPIC)
#
#         Youngs_Modulus Poissons_Ratio Shear_Modulus Mass_Density
#         Thermal_Exp_Coeff Thermal_Cond Plastic_Strain_Ratio
#         Work_Hardening Forming_Limit Stress_Strain
#         Yield Ultimate_Tensile
#         Specific_Heat
#         Fatigue_Strength_Coef Fatigue_Strength_Exp
#         Fatigue_Ductility_Coef Fatigue_Ductility_Exp
#         Latent_Heat Phase_Temp Phase_Temp_Range Phase_Specific_Heat
#         IR_Scatter IR_Extinct Solar_Scatter Solar_Extinct
#         Elec_Resitivity
#         Max_Stress_Tension Max_Stress_Compression Max_Shear_Stress
#         Max_Strain_Tension Max_Strain_Compression Max_Shear_Strain
#         Tsai_Wu Damping_Coeff
#
# When Type == ORTHO (ORTHOTROPIC)
#
#         Youngs_Modulus_X Youngs_Modulus_Y Youngs_Modulus_Z
#         Poissons_Ratio_X Poissons_Ratio_Y Poissons_Ratio_Z
#         Shear_Modulus_X Shear_Modulus_Y Shear_Modulus_Z Mass_Density
#         Thermal_Exp_Coeff_X Thermal_Exp_Coeff_Y Thermal_Exp_Coeff_Z
#         Thermal_Cond_X Thermal_Cond_Y Thermal_Cond_Z
#         Specific_Heat
#         Latent_Heat Phase_Temp Phase_Temp_Range Phase_Specific_Heat
#         IR_Scatter IR_Extinct Solar_Scatter Solar_Extinct
#         Elec_Resitivity
#         Max_Stress_Tension_X Max_Stress_Tension_Y Max_Stress_Tension_Z
#         Max_Stress_Compression_X Max_Stress_Compression_Y Max_Stress_Compression_Z
#         Max_Shear_Stress
#         Max_Strain_Tension_X Max_Strain_Tension_Y Max_Strain_Tension_Z
#         Max_Strain_Compression_X Max_Strain_Tension_Y Max_Strain_Tension_Z
#         Max_Shear_Strain
#         Tsai_Wu Damping_Coeff
#
# When Type == ANISO (ANISOTROPIC)
#
#         G_1_1
#         G_2_1 G_2_2
#         G_3_1 G_3_2 G_3_3
#         G_4_1 G_4_2 G_4_3 G_4_4
#         G_5_1 G_5_2 G_5_3 G_5_4 G_5_5
#         G_6_1 G_6_2 G_6_3 G_6_4 G_6_5 G_6_6
#         Mass_Density
#         Thermal_Exp_Coeff_1 Thermal_Exp_Coeff_2 Thermal_Exp_Coeff_3
#         Thermal_Exp_Coeff_4 Thermal_Exp_Coeff_5 Thermal_Exp_Coeff_6
#         Thermal_Cond_1_1
#         Thermal_Cond_2_1 Thermal_Cond_2_2
#         Thermal_Cond_3_1 Thermal_Cond_3_2 Thermal_Cond_3_3
#         Specific_Heat Damping_Coeff
#
# When Type == FLUID (FLUID)
#
#         Mass_Density Thermal_Cond Thermal_Exp_Coeff
#         Dynamic_Visc Specific_Heat_Pressure Gas_Constant
#         Bulk_Modulus Sound_Speed Damping_Coeff Molar_Mass
#         Version
#
# NOTE: Each of these is a single record in the Material Library file.
#       <\> back slash can be used as delimeter to indicate continuation to
#       next line.
#
#         Libref   is a number string (>0) which is an unique identifier in the
#                  library
#         Name     is a string
#         Category is METAL or
#                     PLASTIC or
#                     OTHER
#         Type     is ISO   (ISOTROPIC) or
#                     ORTHO (ORTHOTROPIC) or
#                     ANISO (ANISOTROPIC) or
#                     FLUID (FLUID)
#         Version is a string.
#         Ref_Temp is a double
#         Plastic_Strain_Ratio is a double
#         Gas_Constant is a double
#         All others are doubles or Temperature Table references.
#
# Temperature Table references may contain either a single scalar
# value, or may contain the name of a temperature table record.
# The temperature table records must begin with the letter 't' or 'T'.
# Each field is delimited by a blank space.
# The name given in the temperature table reference must exactly match
# the name given on the temperature table record. The name must be the
# first token of the temperature table record.
# The second token of the temperature table is the count of the number of
# temperature-value pairs. The remaining tokens are the temperature-value
# pairs.
#
# Use the following convention for table names.
#
# eg: T3Youngs  where:
#
#     T  - every table must start with character "t" or "T"
#     3  - Libref of the material to ensure the table names are unique
#     Youngs - Property name
#
# Table format example:
#
# T3Youngs 3 0.0 23.5 50 1.456e2 100. 2.34e3
# The above table has 3 temperature-value pairs as follows:
#     0.0  - 23.5
#     50   - 1.456e2
#     100. - 2.34e3
#
# The character "#" in the first token position of a record indicates a comment
# record. This can be useful for putting column headings in the database.
#
# The TAB character may be used as a delimiter between tokens in a record.
# This may be useful to vertically align the columns.
#
# If a field of a record is to be not set, code the string "EMPTY" to occupy
# the position of that field.
#
# Queries against string fields (only name at this time) are by default
# case insensitive. This is done in DBC_create_criterion by folding the
# criteria string to upper case and then folding the dataBase field to
# upper case in DBC_execute_query as the strings are compared.
# To escape away from case insensitive queries, that is, to query against
# a specific case of a character, enclose the criteria string in {}.
#
#
# 
#
###############################################################################
#MOM_set_debug_mode on
#---------------------------------------------
# Non DBC Global Variables
#---------------------------------------------
set dbFileName "[MOM_ask_env_var UGII_PHYS_MATERIAL_LIB_DATA_DIR]phys_material.dat"
set lastRowNumber  0
set DEBUG "0"
set TEST "0"

#---------------------------------------------
# Global variables set by DBC for Input/Output
#---------------------------------------------
set dbc_lhs_exp     ""
set dbc_rhs_exp     ""
set dbc_relop       ""
set dbc_query       ""
set dbc_subqry1     ""
set dbc_subqry2     ""
set dbc_boolop      ""
set dbc_class_name  ""
set dbc_attr_count  0
set dbc_attr_id     ""
set dbc_query_count 0
set dbc_libref      ""
set dbc_var_list    ""


#---------------------------------------------
proc DBC_init_db {} {
#---------------------------------------------
   global DEBUG
   global dataBase
   global lastRowNumber
   global dbFileName
   global line

   set fp [open $dbFileName "r"]

   set row 0

   # Read the ASCII database file into a 2-dimensional array 'dataBase'
   while { [DBC_read_full_record $fp] >= 0 } {

      scan $line "%s" first_token
      set first_token [string range $first_token 0 0]

      set dataBase($row,0) " "
      set dataBase($row,1) " "
      set dataBase($row,2) " "
      set dataBase($row,3) " "
      set dataBase($row,4) " "

      if {"$first_token" != "t"       \
       && "$first_token" != "T"       \
       && "$first_token" != "#"}      {
          # Read the first 5 tokens.
          # These are common to all materials.
          scan $line "%s %s %s %s %s" \
              dataBase($row,0) dataBase($row,1) dataBase($row,2) \
              dataBase($row,3) dataBase($row,4)


          }
          # Read the first 5 tokens.

       # Set the rest of the array to blanks.
       # This is needed only because of a bug where all the
       # variables need to be in the rset.
       set dataBase($row,5) " "
       set dataBase($row,6) " "
       set dataBase($row,7) " "
       set dataBase($row,8) " "
       set dataBase($row,9) " "
       set dataBase($row,10) " "
       set dataBase($row,11) " "
       set dataBase($row,12) " "
       set dataBase($row,13) " "
       set dataBase($row,14) " "
       set dataBase($row,15) " "
       set dataBase($row,16) " "
       set dataBase($row,17) " "
       set dataBase($row,18) " "
       set dataBase($row,19) " "
       set dataBase($row,20) " "
       set dataBase($row,21) " "
       set dataBase($row,22) " "
       set dataBase($row,23) " "
       set dataBase($row,24) " "
       set dataBase($row,25) " "
       set dataBase($row,26) " "
       set dataBase($row,27) " "
       set row [expr $row + 1]


   }
   set lastRowNumber [expr $row - 1]

   close $fp

   if { "$DEBUG" == "1" } {
      puts "The database is from file $dbFileName"
      puts "libref    Name    Category    Type        Ref_Temp"

      for {set ix 0} {$ix <= $lastRowNumber} {set ix [expr $ix + 1]} {
         puts "$dataBase($ix,0)    $dataBase($ix,1)    $dataBase($ix,2)    $dataBase($ix,3)    $dataBase($ix,4)"
      }
   }
}

#---------------------------------------------
proc DBC_translate_att_alias {} {
#---------------------------------------------
   global DEBUG
   global dbc_alias
   global dbc_id

   if { "$DEBUG" == "1" } {
      puts "DBC_translate_att_alias(dbc_alias = $dbc_alias, dbc_id = $dbc_id)"
   }

   # The input dbc_id is the column in the RSET
   # The variable db_row is a local variable in execute_query

   set dbc_id "\$dataBase(\$db_row,$dbc_id)"

   if { "$DEBUG" == "1" } {
      puts "DBC_Translate_Att_Alias made: $dbc_id"
   }
}

#---------------------------------------------
proc DBC_create_criterion {} {
#---------------------------------------------
   global DEBUG
   global dbc_lhs_exp
   global dbc_rhs_exp
   global dbc_relop
   global dbc_query
   global fpout


   if { "$DEBUG" == "1" } {
      puts "DBC_create_criterion(dbc_lhs_exp = $dbc_lhs_exp, dbc_rhs_exp = $dbc_rhs_exp, dbc_relop = $dbc_relop)"
   }

   # For now just assume lhs is of form db(alias)
   scan $dbc_lhs_exp "DB(%s" x

   if { "$DEBUG" == "1" } {
      puts "x = $x"
   }
   set y [string last ) $x]
   set y [expr $y - 1]
   set dbc_lhs_exp [string range $x 0 $y]

   set dbc_query " $dbc_lhs_exp $dbc_relop \"$dbc_rhs_exp\" "

   if {$dbc_relop == "=="}  {
       set dbc_command_prefix "\[ string match "

       # Fold the string to upper if the first char is NOT "LEFT-BRACE"
       # Cannot type the actual char here because it gets parsed
       set dcb_initial_char [string index "$dbc_rhs_exp" 0]
       set dcb_toupper " "
       set dcb_closer " "

       if { $dcb_initial_char != "\{" }  {
           set dbc_rhs_exp [string toupper "$dbc_rhs_exp"]
           set dcb_toupper " \[ string toupper"
           set dcb_closer " \]"
       }

       set dbc_query "$dbc_command_prefix \"$dbc_rhs_exp\" $dcb_toupper $dbc_lhs_exp $dcb_closer \]"
   }

   if { "$DEBUG" == "1" } {
      puts "DBC_Create_Criterion made: $dbc_query"
   }
}

#---------------------------------------------
proc DBC_create_query {} {
#---------------------------------------------
   global DEBUG
   global dbc_query
   global dbc_subqry1
   global dbc_subqry2
   global dbc_boolop

   if { "$DEBUG" == "1" } {
      puts "DBC_create_query(dbc_subqry1 = $dbc_subqry1, dbc_subqry2 = $dbc_subqry2, dbc_boolop = $dbc_boolop)"
   }

   set dbc_query "$dbc_subqry1 $dbc_boolop $dbc_subqry2"

   if { "$DEBUG" == "1" } {
      puts "DBC_Create_Query made: $dbc_query"

   set $dbc_subqry1 " "
   set $dbc_subqry2 " "

   }
}

#---------------------------------------------
proc DBC_execute_query {} {
#---------------------------------------------
   global DEBUG
   global dbc_class_name
   global dbc_query
   global dbc_attr_count
   global dbc_attr_aliases
   global dbc_attr_id
   global lastRowNumber
   global dataBase

   if { "$DEBUG" == "1" } {
      puts "DBC_execute_query(dbc_class_name = $dbc_class_name, dbc_query = $dbc_query)"
   }

   DBC_load_rset_data # Load the structure of the RSET

   set rset_row 0

   for {set db_row 0} {$db_row <= $lastRowNumber} {incr db_row 1} {
      if { [expr $dbc_query]  } {
         for {set ix 0} {$ix < $dbc_attr_count} { incr ix 1 } {
            DBC_ask_attr_id $dbc_attr_aliases($ix)
            DBC_write_to_rset $rset_row $dbc_attr_aliases($ix) \
                                $dataBase($db_row,$dbc_attr_id)
         }
         incr rset_row 1
      }
   }

   if { "$DEBUG" == "1" } {
      puts "DBC_Execute_Query: Query $dbc_query, RSET rows: $rset_row"
   }
}

#---------------------------------------------
proc DBC_execute_query_for_count {} {
#---------------------------------------------
   global DEBUG
   global dbc_class_name
   global dbc_query
   global dbc_query_count
   global lastRowNumber
   global dataBase

   if { "$DEBUG" == "1" } {
      puts "DBC_execute_query_for_count(dbc_class_name = $dbc_class_name, dbc_query = $dbc_query)"
   }

   set dbc_query_count 0

   for {set db_row 0} {$db_row <= $lastRowNumber} {incr db_row 1} {
      if { [expr $dbc_query]  } {
         incr dbc_query_count 1
      }
   }

   if { "$DEBUG" == "1" } {
      puts "DBC_Execute_Query_For_Count: Query $dbc_query, Count: $dbc_query_count"
   }

   set dbc_query " "
}


#---------------------------------------------
proc DBC_retrieve {} {
#---------------------------------------------
    global DEBUG
    global TEST

    global dbc_libref
    global dbc_retrieve_var_list
    global lastRowNumber
    global dataBase
    global dbFileName

    # These variables were documented by the matl_lib application.
    # The matl_lib UG Object will be asking for these values.

    # Common to all materials.
    global dbc_Name
    global dbc_Category
    global dbc_Type
    global dbc_Ref_Temp
    global dbc_Version

    # Isotropic Materials
    global dbc_Youngs_Modulus
    global dbc_Poissons_Ratio
    global dbc_Shear_Modulus
    global dbc_Mass_Density
    global dbc_Thermal_Exp_Coeff
    global dbc_Thermal_Cond
    global dbc_Plastic_Strain_Ratio
    global dbc_Work_Hardening
    global dbc_Forming_Limit
    global dbc_Stress_Strain
    global dbc_Yield
    global dbc_Ultimate_Tensile
    global dbc_Specific_Heat
    global dbc_Fatigue_Strength_Coef
    global dbc_Fatigue_Strength_Exp
    global dbc_Fatigue_Ductility_Coef
    global dbc_Fatigue_Ductility_Exp
    global dbc_Latent_Heat
    global dbc_Phase_Temp
    global dbc_Phase_Temp_Range
    global dbc_Phase_Specific_Heat
    global dbc_IR_Scatter
    global dbc_IR_Extinct
    global dbc_Solar_Scatter
    global dbc_Solar_Extinct
    global dbc_Elec_Resitivity
    global dbc_Max_Stress_Tension
    global dbc_Max_Stress_Compression
    global dbc_Max_Shear_Stress
    global dbc_Max_Strain_Tension
    global dbc_Max_Strain_Compression
    global dbc_Max_Shear_Strain
    global dbc_Tsai_Wu

    # Orthotropic Materials
    global dbc_Youngs_Modulus_X
    global dbc_Youngs_Modulus_Y
    global dbc_Youngs_Modulus_Z
    global dbc_Poissons_ratio_X
    global dbc_Poissons_ratio_Y
    global dbc_Poissons_ratio_Z
    global dbc_Shear_modulus_X
    global dbc_Shear_modulus_Y
    global dbc_Shear_modulus_Z
    global dbc_Thermal_Exp_Coeff_X
    global dbc_Thermal_Exp_Coeff_Y
    global dbc_Thermal_Exp_Coeff_Z
    global dbc_Thermal_Cond_X
    global dbc_Thermal_Cond_Y
    global dbc_Thermal_Cond_Z
    global dbc_Max_Stress_Tension_X
    global dbc_Max_Stress_Tension_Y
    global dbc_Max_Stress_Tension_Z
    global dbc_Max_Stress_Compression_X
    global dbc_Max_Stress_Compression_Y
    global dbc_Max_Stress_Compression_Z
    global dbc_Max_Strain_Tension_X
    global dbc_Max_Strain_Tension_Y
    global dbc_Max_Strain_Tension_Z
    global dbc_Max_Strain_Compression_X
    global dbc_Max_Strain_Compression_Y
    global dbc_Max_Strain_Compression_Z

    # Anisotropic Materials
    global dbc_G_1_1
    global dbc_G_2_1
    global dbc_G_2_2
    global dbc_G_3_1
    global dbc_G_3_2
    global dbc_G_3_3
    global dbc_G_4_1
    global dbc_G_4_2
    global dbc_G_4_3
    global dbc_G_4_4
    global dbc_G_5_1
    global dbc_G_5_2
    global dbc_G_5_3
    global dbc_G_5_4
    global dbc_G_5_5
    global dbc_G_6_1
    global dbc_G_6_2
    global dbc_G_6_3
    global dbc_G_6_4
    global dbc_G_6_5
    global dbc_G_6_6
    global dbc_Thermal_Exp_Coeff_1
    global dbc_Thermal_Exp_Coeff_2
    global dbc_Thermal_Exp_Coeff_3
    global dbc_Thermal_Exp_Coeff_4
    global dbc_Thermal_Exp_Coeff_5
    global dbc_Thermal_Exp_Coeff_6
    global dbc_Thermal_Cond_1_1
    global dbc_Thermal_Cond_2_1
    global dbc_Thermal_Cond_2_2
    global dbc_Thermal_Cond_3_1
    global dbc_Thermal_Cond_3_2
    global dbc_Thermal_Cond_3_3

    # Fluid Materials
    global dbc_Dynamic_Visc;
    global dbc_Specific_Heat_Pressure;
    global dbc_Gas_Constant;
    global dbc_Bulk_Modulus;
    global dbc_Sound_Speed;
    global dbc_Damping_Coeff;
    global dbc_Molar_Mass;

    global line
    global table_string
    global second_line

    set dat_file_version 0

    if { "$DEBUG" == "1" || "$TEST" == "1" }  \
    {
        puts "DBC_retrieve(libref = $dbc_libref)"
        puts "DBC_retrieve(lastRowNumber = $lastRowNumber)"
    }

    set fp [open $dbFileName "r"]

    for {set db_row 0} {$db_row <= $lastRowNumber} {incr db_row 1} {

        # Search for the correct record.
        # For each row search, step through the file at the same time.

        DBC_read_full_record $fp

        scan $line "%s %s %s %s %s"     \
              first_token               \
              dbc_Name                  \
              dbc_Category              \
              dbc_Type                  \
              dbc_Ref_Temp

        if { $dbc_libref == $first_token }   {
            #  Find the row for the desired LibRef

            if { "$DEBUG" == "1" || "$TEST" == "1" }  {
                puts "Found libref in row $db_row"
                puts "$dbc_Name"
                puts "$dbc_Category"
                puts "Material Type is $dbc_Type"
            }

             break
             # Stop the search loop.
        }
        #  Find the row for the desired LibRef
    }
    # Search for the correct record.

    close $fp

    # Set the variables based upon the values of this row
    # (for the desired LibRef)
    switch $dbc_Type  {
        ISO  {

            set  dat_file_version 0
            set  second_line $line
            # Version 1
            set  dbc_Ref_Temp "EMPTY"
            set  dbc_Youngs_Modulus "EMPTY"
            set  dbc_Poissons_Ratio "EMPTY"
            set  dbc_Shear_Modulus "EMPTY"
            set  dbc_Mass_Density "EMPTY"
            set  dbc_Thermal_Exp_Coeff "EMPTY"
            set  dbc_Thermal_Cond "EMPTY"
            set  dbc_Plastic_Strain_Ratio "EMPTY"
            set  dbc_Yield "EMPTY"
            set  dbc_Ultimate_Tensile "EMPTY"
            set  dbc_Work_Hardening "EMPTY"
            set  dbc_Forming_Limit "EMPTY"
            set  dbc_Stress_Strain "EMPTY"
            # Version 2
            set  dbc_Specific_Heat "EMPTY"
            set  dbc_Fatigue_Strength_Coef "EMPTY"
            set  dbc_Fatigue_Strength_Exp "EMPTY"
            set  dbc_Fatigue_Ductility_Coef "EMPTY"
            set  dbc_Fatigue_Ductility_Exp "EMPTY"
            # Version 3
            set  dbc_Latent_Heat "EMPTY"
            set  dbc_Phase_Temp "EMPTY"
            set  dbc_Phase_Temp_Range "EMPTY"
            set  dbc_Phase_Specific_Heat "EMPTY"
            set  dbc_IR_Scatter "EMPTY"
            set  dbc_IR_Extinct "EMPTY"
            set  dbc_Solar_Scatter "EMPTY"
            set  dbc_Solar_Extinct "EMPTY"
            set  dbc_Elec_Resitivity "EMPTY"
            set  dbc_Max_Stress_Tension "EMPTY"
            set  dbc_Max_Stress_Compression "EMPTY"
            set  dbc_Max_Shear_Stress "EMPTY"
            set  dbc_Max_Strain_Tension "EMPTY"
            set  dbc_Max_Strain_Compression "EMPTY"
            set  dbc_Max_Shear_Strain "EMPTY"
            set  dbc_Tsai_Wu "EMPTY"
            # Version 4
            set  dbc_Damping_Coeff "EMPTY"
            set  dbc_Version "EMPTY"

            # Version 1
            DBC_scan_and_strip
            set dbc_libref $table_string
            DBC_scan_and_strip
            set dbc_Name $table_string
            DBC_scan_and_strip
            set dbc_Category $table_string
            DBC_scan_and_strip
            set dbc_Type $table_string
            DBC_scan_and_strip
            set dbc_Ref_Temp $table_string
            DBC_scan_and_strip
            set dbc_Youngs_Modulus $table_string
            DBC_scan_and_strip
            set dbc_Poissons_Ratio  $table_string
            DBC_scan_and_strip
            set dbc_Shear_Modulus  $table_string
            DBC_scan_and_strip
            set dbc_Mass_Density  $table_string
            DBC_scan_and_strip
            set dbc_Thermal_Exp_Coeff  $table_string
            DBC_scan_and_strip
            set dbc_Thermal_Cond  $table_string
            DBC_scan_and_strip
            set dbc_Plastic_Strain_Ratio  $table_string
            DBC_scan_and_strip
            set dbc_Yield  $table_string
            DBC_scan_and_strip
            set dbc_Ultimate_Tensile  $table_string
            DBC_scan_and_strip
            set dbc_Work_Hardening  $table_string
            DBC_scan_and_strip
            set dbc_Forming_Limit  $table_string
            DBC_scan_and_strip
            set dbc_Stress_Strain  $table_string

            # Version 2
            DBC_scan_and_strip
            set dbc_Specific_Heat  $table_string
            DBC_scan_and_strip
            set dbc_Version  $table_string
            if { $dbc_Version == "NONE" } {
                set  dbc_Version $dbc_Specific_Heat
                set  dbc_Specific_Heat "EMPTY"
                set  dat_file_version 1
            } else {
                set dbc_Fatigue_Strength_Coef $dbc_Version
                DBC_scan_and_strip
                set dbc_Fatigue_Strength_Exp  $table_string
                DBC_scan_and_strip
                set dbc_Fatigue_Ductility_Coef  $table_string
                DBC_scan_and_strip
                set dbc_Fatigue_Ductility_Exp  $table_string

                # Version 3
                DBC_scan_and_strip
                set dbc_Latent_Heat  $table_string
                DBC_scan_and_strip
                set dbc_Version  $table_string
                if { $dbc_Version == "NONE" } {
                    set dbc_Version $dbc_Latent_Heat
                    set dbc_Latent_Heat "EMPTY"
                    set  dat_file_version 2
                } else {
                    set dbc_Phase_Temp $dbc_Version
                    DBC_scan_and_strip
                    set dbc_Phase_Temp_Range  $table_string
                    DBC_scan_and_strip
                    set dbc_Phase_Specific_Heat  $table_string
                    DBC_scan_and_strip
                    set dbc_IR_Scatter  $table_string
                    DBC_scan_and_strip
                    set dbc_IR_Extinct  $table_string
                    DBC_scan_and_strip
                    set dbc_Solar_Scatter  $table_string
                    DBC_scan_and_strip
                    set dbc_Solar_Extinct  $table_string
                    DBC_scan_and_strip
                    set dbc_Elec_Resitivity  $table_string
                    DBC_scan_and_strip
                    set  dbc_Max_Stress_Tension $table_string
                    DBC_scan_and_strip
                    set  dbc_Max_Stress_Compression $table_string
                    DBC_scan_and_strip
                    set  dbc_Max_Shear_Stress $table_string
                    DBC_scan_and_strip
                    set  dbc_Max_Strain_Tension $table_string
                    DBC_scan_and_strip
                    set  dbc_Max_Strain_Compression $table_string
                    DBC_scan_and_strip
                    set  dbc_Max_Shear_Strain $table_string
                    DBC_scan_and_strip
                    set  dbc_Tsai_Wu $table_string

                    # Version 4
                    DBC_scan_and_strip
                    set dbc_Damping_Coeff  $table_string
                    DBC_scan_and_strip
                    set dbc_Version  $table_string
                    if { $dbc_Version == "NONE" } {
                        set dbc_Version $dbc_Damping_Coeff
                        set dbc_Damping_Coeff "EMPTY"
                        set dat_file_version 3
                    } else {
                        set dat_file_version 4
                    }
                }
            }

            if { "$DEBUG" == "1" || "$TEST" == "1" }  {
                puts "libref                  $dbc_libref"
                puts "dbc_Name                $dbc_Name"
                puts "dbc_Category            $dbc_Category"
                puts "dbc_Type                $dbc_Type"
                puts "dbc_Ref_Temp            $dbc_Ref_Temp"
                puts "Youngs_Modulus          $dbc_Youngs_Modulus"
                puts "Poissons_Ratio          $dbc_Poissons_Ratio"
                puts "Shear_Modulus           $dbc_Shear_Modulus"
                puts "Mass_Density            $dbc_Mass_Density"
                puts "Thermal_Exp_Coeff       $dbc_Thermal_Exp_Coeff"
                puts "Thermal_Cond            $dbc_Thermal_Cond"
                puts "Plastic_Strain_Ratio    $dbc_Plastic_Strain_Ratio"
                puts "Yield                   $dbc_Yield"
                puts "Ultimate_Tensile        $dbc_Ultimate_Tensile"
                puts "Work_Hardening          $dbc_Work_Hardening"
                puts "Forming_Limit           $dbc_Forming_Limit"
                puts "Stress_Strain           $dbc_Stress_Strain"
                puts "Specific_Heat           $dbc_Specific_Heat"
                puts "Fatigue_Strength_Coef   $dbc_Fatigue_Strength_Coef"
                puts "Fatigue_Strength_Exp    $dbc_Fatigue_Strength_Exp"
                puts "Fatigue_Ductility_Coef  $dbc_Fatigue_Ductility_Coef"
                puts "Fatigue_Ductility_Exp   $dbc_Fatigue_Ductility_Exp"
                puts "Latent_Heat             $dbc_Latent_Heat"
                puts "Phase_Temp              $dbc_Phase_Temp"
                puts "Phase_Temp_Range        $dbc_Phase_Temp_Range"
                puts "Phase_Specific_Heat     $dbc_Phase_Specific_Heat"
                puts "IR_Scatter              $dbc_IR_Scatter"
                puts "IR_Extinct              $dbc_IR_Extinct"
                puts "Solar_Scatter           $dbc_Solar_Scatter"
                puts "Solar_Extinct           $dbc_Solar_Extinct"
                puts "Elec_Resitivity         $dbc_Elec_Resitivity"
                puts "Max_Stress_Tension      $dbc_Max_Stress_Tension"
                puts "Max_Stress_Compression  $dbc_Max_Stress_Compression"
                puts "Max_Shear_Stress        $dbc_Max_Shear_Stress"
                puts "Max_Strain_Tension      $dbc_Max_Strain_Tension"
                puts "Max_Strain_Compression  $dbc_Max_Strain_Compression"
                puts "Max_Shear_Strain        $dbc_Max_Shear_Strain"
                puts "Tsai_Wu                 $dbc_Tsai_Wu"
                puts "Damping_Coeff           $dbc_Damping_Coeff"
                puts "Version                 $dbc_Version"
                puts "dat_file_version        $dat_file_version"
            }

            set table_string $dbc_Youngs_Modulus
            DBC_read_temperature_table $table_string
            set dbc_Youngs_Modulus $table_string

            set table_string $dbc_Poissons_Ratio
            DBC_read_temperature_table $table_string
            set dbc_Poissons_Ratio $table_string

            set table_string $dbc_Shear_Modulus
            DBC_read_temperature_table $table_string
            set dbc_Shear_Modulus $table_string

            set table_string $dbc_Mass_Density
            DBC_read_temperature_table $table_string
            set dbc_Mass_Density $table_string

            set table_string $dbc_Thermal_Exp_Coeff
            DBC_read_temperature_table $table_string
            set dbc_Thermal_Exp_Coeff $table_string

            set table_string $dbc_Thermal_Cond
            DBC_read_temperature_table $table_string
            set dbc_Thermal_Cond $table_string

            set table_string $dbc_Forming_Limit
            DBC_read_temperature_table $table_string
            set dbc_Forming_Limit $table_string

            set table_string $dbc_Stress_Strain
            DBC_read_temperature_table $table_string
            set dbc_Stress_Strain $table_string

            set table_string $dbc_Yield
            DBC_read_temperature_table $table_string
            set dbc_Yield $table_string

            set table_string $dbc_Ultimate_Tensile
            DBC_read_temperature_table $table_string
            set dbc_Ultimate_Tensile $table_string

            if { $dat_file_version > 1 } {
                set table_string $dbc_Specific_Heat
                DBC_read_temperature_table $table_string
                set dbc_Specific_Heat $table_string
            }

            if { $dat_file_version > 2 } {
                set table_string $dbc_Latent_Heat
                DBC_read_temperature_table $table_string
                set dbc_Latent_Heat $table_string

                set table_string $dbc_Phase_Temp
                DBC_read_temperature_table $table_string
                set dbc_Phase_Temp $table_string

                set table_string $dbc_Phase_Temp_Range
                DBC_read_temperature_table $table_string
                set dbc_Phase_Temp_Range $table_string

                set table_string $dbc_Phase_Specific_Heat
                DBC_read_temperature_table $table_string
                set dbc_Phase_Specific_Heat $table_string

                set table_string $dbc_IR_Scatter
                DBC_read_temperature_table $table_string
                set dbc_IR_Scatter$ table_string

                set table_string $dbc_IR_Extinct
                DBC_read_temperature_table $table_string
                set dbc_IR_Extinct$ table_string

                set table_string $dbc_Solar_Scatter
                DBC_read_temperature_table $table_string
                set dbc_Solar_Scatter $table_string

                set table_string $dbc_Solar_Extinct
                DBC_read_temperature_table $table_string
                set dbc_Solar_Extinct $table_string

                set table_string $dbc_Elec_Resitivity
                DBC_read_temperature_table $table_string
                set dbc_Elec_Resitivity $table_string
            }

            if { $dat_file_version > 3 } {
                set table_string $dbc_Damping_Coeff
                DBC_read_temperature_table $table_string
                set dbc_Damping_Coeff $table_string
            }
        }
        # Material type ISO
        ORTHO {

            # The Tcl interpreter that we are using seems to limit the
            # number of tokens in the scan to 20.
            # Since the Ortho material has 21, split it into 2 strings.
            # Look for the Ref_Temp string. We already have read this.
            # And it is the first scalar value in the record. It will be
            # unique within the first 5 tokens.

            # mraz - fixed bug here; can't use the first occurance of the ref_temp string
            # since it is possible for that string to be a substring in one of the preceding fields
            # use the existing DBC_scan_and_strip call to remove the vars from the front
            set second_line $line

            set dat_file_version 0
            # Version 1
            set dbc_Ref_Temp  "EMPTY"
            set dbc_Youngs_Modulus_X  "EMPTY"
            set dbc_Youngs_Modulus_Y  "EMPTY"
            set dbc_Youngs_Modulus_Z  "EMPTY"
            set dbc_Poissons_ratio_X  "EMPTY"
            set dbc_Poissons_ratio_Y  "EMPTY"
            set dbc_Poissons_ratio_Z  "EMPTY"
            set dbc_Shear_modulus_X  "EMPTY"
            set dbc_Shear_modulus_Y  "EMPTY"
            set dbc_Shear_modulus_Z  "EMPTY"
            set dbc_Mass_Density  "EMPTY"
            set dbc_Thermal_Exp_Coeff_X  "EMPTY"
            set dbc_Thermal_Exp_Coeff_Y  "EMPTY"
            set dbc_Thermal_Exp_Coeff_Z  "EMPTY"
            set dbc_Thermal_Cond_X  "EMPTY"
            set dbc_Thermal_Cond_Y  "EMPTY"
            set dbc_Thermal_Cond_Z  "EMPTY"
            # Version 2
            set dbc_Specific_Heat  "EMPTY"
            # Version 3
            set dbc_Latent_Heat  "EMPTY"
            set dbc_Phase_Temp  "EMPTY"
            set dbc_Phase_Temp_Range  "EMPTY"
            set dbc_Phase_Specific_Heat  "EMPTY"
            set dbc_IR_Scatter  "EMPTY"
            set dbc_IR_Extinct  "EMPTY"
            set dbc_Solar_Scatter  "EMPTY"
            set dbc_Solar_Extinct  "EMPTY"
            set dbc_Elec_Resitivity  "EMPTY"
            set dbc_Max_Stress_Tension_X  "EMPTY"
            set dbc_Max_Stress_Tension_Y  "EMPTY"
            set dbc_Max_Stress_Tension_Z  "EMPTY"
            set dbc_Max_Stress_Compression_X  "EMPTY"
            set dbc_Max_Stress_Compression_Y  "EMPTY"
            set dbc_Max_Stress_Compression_Z  "EMPTY"
            set dbc_Max_Shear_Stress  "EMPTY"
            set dbc_Max_Strain_Tension_X  "EMPTY"
            set dbc_Max_Strain_Tension_Y  "EMPTY"
            set dbc_Max_Strain_Tension_Z  "EMPTY"
            set dbc_Max_Strain_Compression_X  "EMPTY"
            set dbc_Max_Strain_Compression_Y  "EMPTY"
            set dbc_Max_Strain_Compression_Z  "EMPTY"
            set dbc_Max_Shear_Strain  "EMPTY"
            set dbc_Tsai_Wu  "EMPTY"
            # Version 4
            set dbc_Damping_Coeff  "EMPTY"

            DBC_scan_and_strip
            set dbc_libref $table_string
            DBC_scan_and_strip
            set dbc_Name $table_string
            DBC_scan_and_strip
            set dbc_Category $table_string
            DBC_scan_and_strip
            set dbc_Type $table_string
            # Version 1
            DBC_scan_and_strip
            set dbc_Ref_Temp $table_string
            DBC_scan_and_strip
            set dbc_Youngs_Modulus_X $table_string
            DBC_scan_and_strip
            set dbc_Youngs_Modulus_Y $table_string
            DBC_scan_and_strip
            set dbc_Youngs_Modulus_Z $table_string
            DBC_scan_and_strip
            set dbc_Poissons_ratio_X $table_string
            DBC_scan_and_strip
            set dbc_Poissons_ratio_Y $table_string
            DBC_scan_and_strip
            set dbc_Poissons_ratio_Z $table_string
            DBC_scan_and_strip
            set dbc_Shear_modulus_X $table_string
            DBC_scan_and_strip
            set dbc_Shear_modulus_Y $table_string
            DBC_scan_and_strip
            set dbc_Shear_modulus_Z $table_string
            DBC_scan_and_strip
            set dbc_Mass_Density $table_string
            DBC_scan_and_strip
            set dbc_Thermal_Exp_Coeff_X $table_string
            DBC_scan_and_strip
            set dbc_Thermal_Exp_Coeff_Y $table_string
            DBC_scan_and_strip
            set dbc_Thermal_Exp_Coeff_Z $table_string
            DBC_scan_and_strip
            set dbc_Thermal_Cond_X $table_string
            DBC_scan_and_strip
            set dbc_Thermal_Cond_Y $table_string
            DBC_scan_and_strip
            set dbc_Thermal_Cond_Z $table_string

            # Version 2
            DBC_scan_and_strip
            set dbc_Specific_Heat $table_string
            DBC_scan_and_strip
            set dbc_Version $table_string
            if { $dbc_Version == "NONE" } {
                set dbc_Version $dbc_Specific_Heat
                set dbc_Specific_Heat "EMPTY"
                set dat_file_version 1
            } else {

                # Version 3
                set dbc_Latent_Heat $dbc_Version
                DBC_scan_and_strip
                set dbc_Version $table_string
                if { $dbc_Version == "NONE" } {
                    set dbc_Version $dbc_Latent_Heat
                    set dbc_Latent_Heat "EMPTY"
                    set dat_file_version 2
                } else {
                    set dbc_Phase_Temp $dbc_Version
                    DBC_scan_and_strip
                    set dbc_Phase_Temp_Range $table_string
                    DBC_scan_and_strip
                    set dbc_Phase_Specific_Heat $table_string
                    DBC_scan_and_strip
                    set dbc_IR_Scatter $table_string
                    DBC_scan_and_strip
                    set dbc_IR_Extinct $table_string
                    DBC_scan_and_strip
                    set dbc_Solar_Scatter $table_string
                    DBC_scan_and_strip
                    set dbc_Solar_Extinct $table_string
                    DBC_scan_and_strip
                    set dbc_Elec_Resitivity $table_string
                    DBC_scan_and_strip
                    set dbc_Max_Stress_Tension_X $table_string
                    DBC_scan_and_strip
                    set dbc_Max_Stress_Tension_Y $table_string
                    DBC_scan_and_strip
                    set dbc_Max_Stress_Tension_Z $table_string
                    DBC_scan_and_strip
                    set dbc_Max_Stress_Compression_X $table_string
                    DBC_scan_and_strip
                    set dbc_Max_Stress_Compression_Y $table_string
                    DBC_scan_and_strip
                    set dbc_Max_Stress_Compression_Z $table_string
                    DBC_scan_and_strip
                    set dbc_Max_Shear_Stress $table_string
                    DBC_scan_and_strip
                    set dbc_Max_Strain_Tension_X $table_string
                    DBC_scan_and_strip
                    set dbc_Max_Strain_Tension_Y $table_string
                    DBC_scan_and_strip
                    set dbc_Max_Strain_Tension_Z $table_string
                    DBC_scan_and_strip
                    set dbc_Max_Strain_Compression_X $table_string
                    DBC_scan_and_strip
                    set dbc_Max_Strain_Compression_Y $table_string
                    DBC_scan_and_strip
                    set dbc_Max_Strain_Compression_Z $table_string
                    DBC_scan_and_strip
                    set dbc_Max_Shear_Strain $table_string
                    DBC_scan_and_strip
                    set dbc_Tsai_Wu $table_string

                    # Version 4
                    DBC_scan_and_strip
                    set dbc_Damping_Coeff $table_string
                    DBC_scan_and_strip
                    set dbc_Version $table_string
                    if { $dbc_Version == "NONE" } {
                        set dbc_Version $dbc_Damping_Coeff
                        set dbc_Damping_Coeff "EMPTY"
                        set dat_file_version 3
                    } else {
                        set dat_file_version 4
                    }
                }
            }

            if { "$DEBUG" == "1" || "$TEST" == "1" }  {
                puts "dbc_Ref_Temp                   $dbc_Ref_Temp"
                puts "dbc_Youngs_Modulus_X           $dbc_Youngs_Modulus_X"
                puts "dbc_Youngs_Modulus_Y           $dbc_Youngs_Modulus_Y"
                puts "dbc_Youngs_Modulus_Z           $dbc_Youngs_Modulus_Z"
                puts "dbc_Poissons_ratio_X           $dbc_Poissons_ratio_X"
                puts "dbc_Poissons_ratio_Y           $dbc_Poissons_ratio_Y"
                puts "dbc_Poissons_ratio_Z           $dbc_Poissons_ratio_Z"
                puts "dbc_Shear_modulus_X            $dbc_Shear_modulus_X"
                puts "dbc_Shear_modulus_Y            $dbc_Shear_modulus_Y"
                puts "dbc_Shear_modulus_Z            $dbc_Shear_modulus_Z"
                puts "dbc_Mass_Density               $dbc_Mass_Density"
                puts "dbc_Thermal_Exp_Coeff_X        $dbc_Thermal_Exp_Coeff_X"
                puts "dbc_Thermal_Exp_Coeff_Y        $dbc_Thermal_Exp_Coeff_Y"
                puts "dbc_Thermal_Exp_Coeff_Z        $dbc_Thermal_Exp_Coeff_Z"
                puts "dbc_Thermal_Cond_X             $dbc_Thermal_Cond_X"
                puts "dbc_Thermal_Cond_Y             $dbc_Thermal_Cond_Y"
                puts "dbc_Thermal_Cond_Z             $dbc_Thermal_Cond_Z"
                puts "dbc_Specific_Heat              $dbc_Specific_Heat"
                puts "dbc_Latent_Heat                $dbc_Latent_Heat"
                puts "dbc_Phase_Temp                 $dbc_Phase_Temp"
                puts "dbc_Phase_Temp_Range           $dbc_Phase_Temp_Range"
                puts "dbc_Phase_Specific_Heat        $dbc_Phase_Specific_Heat"
                puts "dbc_IR_Scatter                 $dbc_IR_Scatter"
                puts "dbc_IR_Extinct                 $dbc_IR_Extinct"
                puts "dbc_Solar_Scatter              $dbc_Solar_Scatter"
                puts "dbc_Solar_Extinct              $dbc_Solar_Extinct"
                puts "dbc_Elec_Resitivity            $dbc_Elec_Resitivity"
                puts "dbc_Max_Stress_Tension_X       $dbc_Max_Stress_Tension_X"
                puts "dbc_Max_Stress_Tension_Y       $dbc_Max_Stress_Tension_Y"
                puts "dbc_Max_Stress_Tension_Z       $dbc_Max_Stress_Tension_Z"
                puts "dbc_Max_Stress_Compression_X   $dbc_Max_Stress_Compression_X"
                puts "dbc_Max_Stress_Compression_Y   $dbc_Max_Stress_Compression_Y"
                puts "dbc_Max_Stress_Compression_Z   $dbc_Max_Stress_Compression_Z"
                puts "dbc_Max_Shear_Stress           $dbc_Max_Shear_Stress"
                puts "dbc_Max_Strain_Tension_X       $dbc_Max_Strain_Tension_X"
                puts "dbc_Max_Strain_Tension_Y       $dbc_Max_Strain_Tension_Y"
                puts "dbc_Max_Strain_Tension_Z       $dbc_Max_Strain_Tension_Z"
                puts "dbc_Max_Strain_Compression_X   $dbc_Max_Strain_Compression_X"
                puts "dbc_Max_Strain_Compression_Y   $dbc_Max_Strain_Compression_Y"
                puts "dbc_Max_Strain_Compression_Z   $dbc_Max_Strain_Compression_Z"
                puts "dbc_Max_Shear_Strain           $dbc_Max_Shear_Strain"
                puts "dbc_Tsai_Wu                    $dbc_Tsai_Wu"
                puts "dbc_Damping_Coeff              $dbc_Damping_Coeff"
                puts "dbc_Version                    $dbc_Version"
                puts "dat_file_version               $dat_file_version"
            }

            set table_string $dbc_Youngs_Modulus_X
            DBC_read_temperature_table $table_string
            set dbc_Youngs_Modulus_X $table_string

            set table_string $dbc_Youngs_Modulus_Y
            DBC_read_temperature_table $table_string
            set dbc_Youngs_Modulus_Y $table_string

            set table_string $dbc_Youngs_Modulus_Z
            DBC_read_temperature_table $table_string
            set dbc_Youngs_Modulus_Z $table_string

            set table_string $dbc_Poissons_ratio_X
            DBC_read_temperature_table $table_string
            set dbc_Poissons_ratio_X $table_string

            set table_string $dbc_Poissons_ratio_Y
            DBC_read_temperature_table $table_string
            set dbc_Poissons_ratio_Y $table_string

            set table_string $dbc_Poissons_ratio_Z
            DBC_read_temperature_table $table_string
            set dbc_Poissons_ratio_Z $table_string

            set table_string $dbc_Shear_modulus_X
            DBC_read_temperature_table $table_string
            set dbc_Shear_modulus_X $table_string

            set table_string $dbc_Shear_modulus_Y
            DBC_read_temperature_table $table_string
            set dbc_Shear_modulus_Y $table_string

            set table_string $dbc_Shear_modulus_Z
            DBC_read_temperature_table $table_string
            set dbc_Shear_modulus_Z $table_string

            set table_string $dbc_Mass_Density
            DBC_read_temperature_table $table_string
            set dbc_Mass_Density $table_string

            set table_string $dbc_Thermal_Exp_Coeff_X
            DBC_read_temperature_table $table_string
            set dbc_Thermal_Exp_Coeff_X $table_string

            set table_string $dbc_Thermal_Exp_Coeff_Y
            DBC_read_temperature_table $table_string
            set dbc_Thermal_Exp_Coeff_Y $table_string

            set table_string $dbc_Thermal_Exp_Coeff_Z
            DBC_read_temperature_table $table_string
            set dbc_Thermal_Exp_Coeff_Z $table_string

            set table_string $dbc_Thermal_Cond_X
            DBC_read_temperature_table $table_string
            set dbc_Thermal_Cond_X $table_string

            set table_string $dbc_Thermal_Cond_Y
            DBC_read_temperature_table $table_string
            set dbc_Thermal_Cond_Y $table_string

            set table_string $dbc_Thermal_Cond_Z
            DBC_read_temperature_table $table_string
            set dbc_Thermal_Cond_Z $table_string

            if { $dat_file_version > 1 } {
                set table_string $dbc_Specific_Heat
                DBC_read_temperature_table $table_string
                set dbc_Specific_Heat $table_string
            }

            if { $dat_file_version > 2 } {
                set table_string $dbc_Latent_Heat
                DBC_read_temperature_table $table_string
                set dbc_Latent_Heat $table_string

                set table_string $dbc_Phase_Temp
                DBC_read_temperature_table $table_string
                set dbc_Phase_Temp $table_string

                set table_string $dbc_Phase_Temp_Range
                DBC_read_temperature_table $table_string
                set dbc_Phase_Temp_Range $table_string

                set table_string $dbc_Phase_Specific_Heat
                DBC_read_temperature_table $table_string
                set dbc_Phase_Specific_Heat $table_string

                set table_string $dbc_IR_Scatter
                DBC_read_temperature_table $table_string
                set dbc_IR_Scatter $table_string

                set table_string $dbc_IR_Extinct
                DBC_read_temperature_table $table_string
                set dbc_IR_Extinct $table_string

                set table_string $dbc_Solar_Scatter
                DBC_read_temperature_table $table_string
                set dbc_Solar_Scatter $table_string

                set table_string $dbc_Solar_Extinct
                DBC_read_temperature_table $table_string
                set dbc_Solar_Extinct $table_string

                set table_string $dbc_Elec_Resitivity
                DBC_read_temperature_table $table_string
                set dbc_Elec_Resitivity $table_string

                set table_string $dbc_Max_Stress_Tension_X
                DBC_read_temperature_table $table_string
                set dbc_Max_Stress_Tension_X $table_string

                set table_string $dbc_Max_Stress_Tension_Y
                DBC_read_temperature_table $table_string
                set dbc_Max_Stress_Tension_Y $table_string

                set table_string $dbc_Max_Stress_Tension_Z
                DBC_read_temperature_table $table_string
                set dbc_Max_Stress_Tension_Z $table_string

                set table_string $dbc_Max_Stress_Compression_X
                DBC_read_temperature_table $table_string
                set dbc_Max_Stress_Compression_X $table_string

                set table_string $dbc_Max_Stress_Compression_Y
                DBC_read_temperature_table $table_string
                set dbc_Max_Stress_Compression_Y $table_string

                set table_string $dbc_Max_Stress_Compression_Z
                DBC_read_temperature_table $table_string
                set dbc_Max_Stress_Compression_Z $table_string

                set table_string $dbc_Max_Shear_Stress
                DBC_read_temperature_table $table_string
                set dbc_Max_Shear_Stress $table_string

                set table_string $dbc_Max_Strain_Tension_X
                DBC_read_temperature_table $table_string
                set dbc_Max_Strain_Tension_X $table_string
                
                set table_string $dbc_Max_Strain_Tension_Y
                DBC_read_temperature_table $table_string 
                set dbc_Max_Strain_Tension_Y $table_string
                
                set table_string $dbc_Max_Strain_Tension_Z
                DBC_read_temperature_table $table_string 
                set dbc_Max_Strain_Tension_Z $table_string
                
                set table_string $dbc_Max_Strain_Compression_X
                DBC_read_temperature_table $table_string 
                set dbc_Max_Strain_Compression_X $table_string
                
                set table_string $dbc_Max_Strain_Compression_Y
                DBC_read_temperature_table $table_string 
                set dbc_Max_Strain_Compression_Y $table_string
                
                set table_string $dbc_Max_Strain_Compression_Z
                DBC_read_temperature_table $table_string 
                set dbc_Max_Strain_Compression_Z $table_string
                
                set table_string $dbc_Max_Shear_Strain
                DBC_read_temperature_table $table_string 
                set dbc_Max_Shear_Strain $table_string               
                
                set table_string $dbc_Tsai_Wu
                DBC_read_temperature_table $table_string 
                set dbc_Tsai_Wu $table_string 
            }

            if { $dat_file_version > 3 } {
                set table_string $dbc_Damping_Coeff
                DBC_read_temperature_table $table_string
                set dbc_Damping_Coeff $table_string
            }

        }
        # Material type ORTHO

        ANISO {

            set second_line $line

            set dat_file_version 0
            # Version 1
            set dbc_Ref_Temp  "EMPTY"
            set dbc_G_1_1  "EMPTY"
            set dbc_G_2_1  "EMPTY"
            set dbc_G_2_2  "EMPTY"
            set dbc_G_3_1  "EMPTY"
            set dbc_G_3_2  "EMPTY"
            set dbc_G_3_3  "EMPTY"
            set dbc_G_4_1  "EMPTY"
            set dbc_G_4_2  "EMPTY"
            set dbc_G_4_3  "EMPTY"
            set dbc_G_4_4  "EMPTY"
            set dbc_G_5_1  "EMPTY"
            set dbc_G_5_2  "EMPTY"
            set dbc_G_5_3  "EMPTY"
            set dbc_G_5_4  "EMPTY"
            set dbc_G_5_5  "EMPTY"
            set dbc_G_6_1  "EMPTY"
            set dbc_G_6_2  "EMPTY"
            set dbc_G_6_3  "EMPTY"
            set dbc_G_6_4  "EMPTY"
            set dbc_G_6_5  "EMPTY"
            set dbc_G_6_6  "EMPTY"
            set dbc_Mass_Density  "EMPTY"
            set dbc_Thermal_Exp_Coeff_1  "EMPTY"
            set dbc_Thermal_Exp_Coeff_2  "EMPTY"
            set dbc_Thermal_Exp_Coeff_3  "EMPTY"
            set dbc_Thermal_Exp_Coeff_4  "EMPTY"
            set dbc_Thermal_Exp_Coeff_5  "EMPTY"
            set dbc_Thermal_Exp_Coeff_6  "EMPTY"
            set dbc_Thermal_Cond_1_1  "EMPTY"
            set dbc_Thermal_Cond_2_1  "EMPTY"
            set dbc_Thermal_Cond_2_2  "EMPTY"
            set dbc_Thermal_Cond_3_1  "EMPTY"
            set dbc_Thermal_Cond_3_2  "EMPTY"
            set dbc_Thermal_Cond_3_3  "EMPTY"
            # Version 2
            set dbc_Specific_Heat  "EMPTY"
            # Version 3
            set dbc_Damping_Coeff  "EMPTY"

            DBC_scan_and_strip
            set dbc_libref $table_string
            DBC_scan_and_strip
            set dbc_Name $table_string
            DBC_scan_and_strip
            set dbc_Category $table_string
            DBC_scan_and_strip
            set dbc_Type $table_string
            # Version 1
            DBC_scan_and_strip
            set dbc_Ref_Temp $table_string
            DBC_scan_and_strip
            DBC_read_temperature_table $table_string
            set dbc_G_1_1 $table_string
            DBC_scan_and_strip
            DBC_read_temperature_table $table_string
            set dbc_G_2_1 $table_string
            DBC_scan_and_strip
            DBC_read_temperature_table $table_string
            set dbc_G_2_2 $table_string
            DBC_scan_and_strip
            DBC_read_temperature_table $table_string
            set dbc_G_3_1 $table_string
            DBC_scan_and_strip
            DBC_read_temperature_table $table_string
            set dbc_G_3_2 $table_string
            DBC_scan_and_strip
            DBC_read_temperature_table $table_string
            set dbc_G_3_3 $table_string
            DBC_scan_and_strip
            DBC_read_temperature_table $table_string
            set dbc_G_4_1 $table_string
            DBC_scan_and_strip
            DBC_read_temperature_table $table_string
            set dbc_G_4_2 $table_string
            DBC_scan_and_strip
            DBC_read_temperature_table $table_string
            set dbc_G_4_3 $table_string
            DBC_scan_and_strip
            DBC_read_temperature_table $table_string
            set dbc_G_4_4 $table_string
            DBC_scan_and_strip
            DBC_read_temperature_table $table_string
            set dbc_G_5_1 $table_string
            DBC_scan_and_strip
            DBC_read_temperature_table $table_string
            set dbc_G_5_2 $table_string
            DBC_scan_and_strip
            DBC_read_temperature_table $table_string
            set dbc_G_5_3 $table_string
            DBC_scan_and_strip
            DBC_read_temperature_table $table_string
            set dbc_G_5_4 $table_string
            DBC_scan_and_strip
            DBC_read_temperature_table $table_string
            set dbc_G_5_5 $table_string
            DBC_scan_and_strip
            DBC_read_temperature_table $table_string
            set dbc_G_6_1 $table_string
            DBC_scan_and_strip
            DBC_read_temperature_table $table_string
            set dbc_G_6_2 $table_string
            DBC_scan_and_strip
            DBC_read_temperature_table $table_string
            set dbc_G_6_3 $table_string
            DBC_scan_and_strip
            DBC_read_temperature_table $table_string
            set dbc_G_6_4 $table_string
            DBC_scan_and_strip
            DBC_read_temperature_table $table_string
            set dbc_G_6_5 $table_string
            DBC_scan_and_strip
            DBC_read_temperature_table $table_string
            set dbc_G_6_6 $table_string
            DBC_scan_and_strip
            DBC_read_temperature_table $table_string
            set dbc_Mass_Density $table_string
            DBC_scan_and_strip
            DBC_read_temperature_table $table_string
            set dbc_Thermal_Exp_Coeff_1 $table_string
            DBC_scan_and_strip
            DBC_read_temperature_table $table_string
            set dbc_Thermal_Exp_Coeff_2 $table_string
            DBC_scan_and_strip
            DBC_read_temperature_table $table_string
            set dbc_Thermal_Exp_Coeff_3 $table_string
            DBC_scan_and_strip
            DBC_read_temperature_table $table_string
            set dbc_Thermal_Exp_Coeff_4 $table_string
            DBC_scan_and_strip
            DBC_read_temperature_table $table_string
            set dbc_Thermal_Exp_Coeff_5 $table_string
            DBC_scan_and_strip
            DBC_read_temperature_table $table_string
            set dbc_Thermal_Exp_Coeff_6 $table_string
            DBC_scan_and_strip
            DBC_read_temperature_table $table_string
            set dbc_Thermal_Cond_1_1 $table_string
            DBC_scan_and_strip
            DBC_read_temperature_table $table_string
            set dbc_Thermal_Cond_2_1 $table_string
            DBC_scan_and_strip
            DBC_read_temperature_table $table_string
            set dbc_Thermal_Cond_2_2 $table_string
            DBC_scan_and_strip
            DBC_read_temperature_table $table_string
            set dbc_Thermal_Cond_3_1 $table_string
            DBC_scan_and_strip
            DBC_read_temperature_table $table_string
            set dbc_Thermal_Cond_3_2 $table_string
            DBC_scan_and_strip
            DBC_read_temperature_table $table_string
            set dbc_Thermal_Cond_3_3 $table_string
            # Version 2
            DBC_scan_and_strip
            set dbc_Specific_Heat $table_string
            DBC_scan_and_strip
            set dbc_Version $table_string
            
            if { $dbc_Version == "NONE" } {
                set dbc_Version $dbc_Specific_Heat
                set dbc_Specific_Heat "EMPTY"
                set dat_file_version 1
            } else {
                # Version 3
                set dbc_Damping_Coeff $dbc_Version
                DBC_scan_and_strip
                set dbc_Version $table_string
                
                if { $dbc_Version == "NONE" } {
                    set dbc_Version $dbc_Damping_Coeff
                    set dbc_Damping_Coeff "EMPTY"
                    set dat_file_version 2
                } else {
                    set dat_file_version 3
                }
            }
            
            if { "$DEBUG" == "1" || "$TEST" == "1" }  {
                puts "dbc_Ref_Temp             $dbc_Ref_Temp"
                puts "dbc_G_1_1                $dbc_G_1_1"
                puts "dbc_G_2_1                $dbc_G_2_1"
                puts "dbc_G_2_2                $dbc_G_2_2"
                puts "dbc_G_3_1                $dbc_G_3_1"
                puts "dbc_G_3_2                $dbc_G_3_2"
                puts "dbc_G_3_3                $dbc_G_3_3"
                puts "dbc_G_4_1                $dbc_G_4_1"
                puts "dbc_G_4_2                $dbc_G_4_2"
                puts "dbc_G_4_3                $dbc_G_4_3"
                puts "dbc_G_4_4                $dbc_G_4_4"
                puts "dbc_G_5_1                $dbc_G_5_1"
                puts "dbc_G_5_2                $dbc_G_5_2"
                puts "dbc_G_5_3                $dbc_G_5_3"
                puts "dbc_G_5_4                $dbc_G_5_4"
                puts "dbc_G_5_5                $dbc_G_5_5"
                puts "dbc_G_6_1                $dbc_G_6_1"
                puts "dbc_G_6_2                $dbc_G_6_2"
                puts "dbc_G_6_3                $dbc_G_6_3"
                puts "dbc_G_6_4                $dbc_G_6_4"
                puts "dbc_G_6_5                $dbc_G_6_5"
                puts "dbc_G_6_6                $dbc_G_6_6"
                puts "dbc_Mass_Density         $dbc_Mass_Density"
                puts "dbc_Thermal_Exp_Coeff_1  $dbc_Thermal_Exp_Coeff_1"
                puts "dbc_Thermal_Exp_Coeff_2  $dbc_Thermal_Exp_Coeff_2"
                puts "dbc_Thermal_Exp_Coeff_3  $dbc_Thermal_Exp_Coeff_3"
                puts "dbc_Thermal_Exp_Coeff_4  $dbc_Thermal_Exp_Coeff_4"
                puts "dbc_Thermal_Exp_Coeff_5  $dbc_Thermal_Exp_Coeff_5"
                puts "dbc_Thermal_Exp_Coeff_6  $dbc_Thermal_Exp_Coeff_6"
                puts "dbc_Thermal_Cond_1_1     $dbc_Thermal_Cond_1_1"
                puts "dbc_Thermal_Cond_2_1     $dbc_Thermal_Cond_2_1"
                puts "dbc_Thermal_Cond_2_2     $dbc_Thermal_Cond_2_2"
                puts "dbc_Thermal_Cond_3_1     $dbc_Thermal_Cond_3_1"
                puts "dbc_Thermal_Cond_3_2     $dbc_Thermal_Cond_3_2"
                puts "dbc_Thermal_Cond_3_3     $dbc_Thermal_Cond_3_3"
                puts "dbc_Specific_Heat        $dbc_Specific_Heat"
                puts "dbc_Damping_Coeff        $dbc_Damping_Coeff"
                puts "dbc_Version              $dbc_Version"
                puts "dat_file_version         $dat_file_version"
            }

            if { $dat_file_version > 1 } {
                set table_string $dbc_Specific_Heat
                DBC_read_temperature_table $table_string
                set dbc_Specific_Heat $table_string
            }

            if { $dat_file_version > 2 } {
                set table_string $dbc_Damping_Coeff
                DBC_read_temperature_table $table_string
                set dbc_Damping_Coeff $table_string
            }
        }
        # Material type ANISO

        FLUID  {
            # The Tcl interpreter that we are using seems to limit the
            # number of tokens in the scan to 20.
            # Split it into 2 strings for processing.
            # Look for the Ref_Temp string. We already have read this.
            # And it is the first scalar value in the record. It will be
            # unique within the first 5 tokens.

            # mraz - fixed bug here; can't use the first occurance of the ref_temp string
            # since it is possible for that string to be a substring in one of the preceding fields
            # use the existing DBC_scan_and_strip call to remove the vars from the front
            set second_line $line

            DBC_scan_and_strip
            set dbc_libref $table_string

            DBC_scan_and_strip
            set dbc_Name $table_string

            DBC_scan_and_strip
            set dbc_Category $table_string

            DBC_scan_and_strip
            set dbc_Type $table_string

            # Version 1
            set  dbc_Mass_Density "EMPTY"
            set  dbc_Thermal_Cond "EMPTY"
            set  dbc_Thermal_Exp_Coeff "EMPTY"
            set  dbc_Dynamic_Visc "EMPTY"
            set  dbc_Specific_Heat_Pressure "EMPTY"
            set  dbc_Gas_Constant "EMPTY"
            # Version 2
            set  dbc_Bulk_Modulus "EMPTY"
            set  dbc_Sound_Speed "EMPTY"
            set  dbc_Damping_Coeff "EMPTY"
            set  dbc_Molar_Mass "EMPTY"
            set  dbc_Version "EMPTY"

            scan $second_line                  \
            "%s %s %s %s %s %s %s %s %s %s %s" \
                dbc_Mass_Density               \
                dbc_Thermal_Cond               \
                dbc_Thermal_Exp_Coeff          \
                dbc_Dynamic_Visc               \
                dbc_Specific_Heat_Pressure     \
                dbc_Gas_Constant               \
                dbc_Bulk_Modulus               \
                dbc_Sound_Speed                \
                dbc_Damping_Coeff              \
                dbc_Molar_Mass                 \
                dbc_Version

            if { "$DEBUG" == "1" || "$TEST" == "1" }  {
                puts "Mass_Density           ::$dbc_Mass_Density"
                puts "Thermal_Cond           ::$dbc_Thermal_Cond"
                puts "Thermal_Exp_Coeff      ::$dbc_Thermal_Exp_Coeff"
                puts "Dynamic_Visc           ::$dbc_Dynamic_Visc"
                puts "Specific_Heat_Pressure ::$dbc_Specific_Heat_Pressure"
                puts "Gas_Constant           ::$dbc_Gas_Constant"
                puts "Bulk_Modulus           ::$dbc_Bulk_Modulus"
                puts "Sound_Speed            ::$dbc_Sound_Speed"
                puts "Damping_Coeff          ::$dbc_Damping_Coeff"
                puts "Molar_Mass             ::$dbc_Molar_Mass"
                puts "Version                ::$dbc_Version"
                puts "dat_file_version       ::$dat_file_version"
            }

            set table_string $dbc_Mass_Density
            DBC_read_temperature_table $table_string
            set dbc_Mass_Density $table_string

            set table_string $dbc_Thermal_Exp_Coeff
            DBC_read_temperature_table $table_string
            set dbc_Thermal_Exp_Coeff $table_string

            set table_string $dbc_Thermal_Cond
            DBC_read_temperature_table $table_string
            set dbc_Thermal_Cond $table_string

            set table_string $dbc_Dynamic_Visc
            DBC_read_temperature_table $table_string
            set dbc_Dynamic_Visc $table_string

            set table_string $dbc_Specific_Heat_Pressure
            DBC_read_temperature_table $table_string
            set dbc_Specific_Heat_Pressure $table_string

            if { $dbc_Version == "NONE" } {
                set dbc_Version $dbc_Bulk_Modulus
                set dbc_Bulk_Modulus "EMPTY"
                set dbc_Sound_Speed "EMPTY"
                set dbc_Damping_Coeff "EMPTY"
                set dbc_Molar_Mass "EMPTY"
            }
        }
        # Material type FLUID
    }
     # for each material type.
}

#---------------------------------------------
proc DBC_read_temperature_table {table_name} {
#---------------------------------------------
   global DEBUG
   global line
   global table_string
   global dbFileName

   if { "$DEBUG" == "1" } {
       puts "DBC_read_temperature_table looking for $table_name"
                          }

   set table_string $table_name

   set first_token [string range $table_name 0 0]

   if {"$first_token" == "T" || "$first_token" == "t"}   {

       set fp [open $dbFileName "r"]
       set row 0

       # Read the ASCII database file for the table_name
       while { [DBC_read_full_record $fp] >= 0 } {

          scan $line "%s" first_token

          if {![string compare $first_token $table_name]}   {
              set table_string $line
              break
          }
       }

       close $fp

   }

   if { "$DEBUG" == "1" } {
      puts "DBC_read_temperature_table: $table_string"
   }

}

#---------------------------------------------
proc DBC_scan_and_strip {} {
#
#  Because this Tcl implementation supports only 20 tokens in scan,
#  This function will scan for the next token,
#  measure its length, then strip the token from the record.
#
#---------------------------------------------
   global DEBUG
   global table_string
   global second_line

   set table_string "NONE"

   scan $second_line "%s" table_string

   set token_len [string length $table_string]

   set second_line [string trimleft $second_line]
   set second_line [string range $second_line [expr $token_len + 1] end]

   if { "$DEBUG" == "1" } {
       puts "DBC_scan_and_strip: table_string = $table_string"
       puts "DBC_scan_and_strip: token_len = $token_len"
       puts "DBC_scan_and_strip: second_line = $second_line"
                          }

}

#---------------------------------------------
proc DBC_read_full_record { fp } {
#
#  Support line continuations in the material records.
#  When reading a record from the material library,
#  Look at the end of the record.
#  If it contains a '\' character, read the next record and
#  append it to the current record.
#
#  Returns the gets status.
#
#---------------------------------------------
   global DEBUG
   global line

   # Keep reading the ASCII database file until there are no more continuations.
   set not_done 1
   set rcode 1
   set line ""
   set continue_index -1

   while { $not_done == 1 && $rcode > 0} {

       set rcode [gets $fp second_line]

       if { $rcode > 0 } {


           set string_len [string length $second_line]
           set continue_index [string last "\\" $second_line]
           incr string_len -2
           incr continue_index -1

           if {$continue_index == $string_len} {

               # Strip the \ from the end
               set second_line [string range $second_line 0 $continue_index]

               set not_done 1

               } else {
                      set not_done 0
                      }
                      # else

           append line $second_line

                            }
                            #  if $rcode
                                           }
                                           #  while


   if { "$DEBUG" == "1" } {
       puts "DBC_read_full_record: "
       puts "DBC_read_full_record:  line = $line"
       puts "DBC_read_full_record:  rcode = $rcode"
                          }
                          # if $DEBUG

return $rcode

}

#########################################################################
##  Added this proc to make it easy to test record retrieval directly
##  in a TCL interpreter.
##  To run/test without any UG, simply comment out "set dbFileName" at
##  the top of the file and add a *new* "set dbFileName" that points to the
##  desired database, then proc test can be run.  Edit test to set the
##  libref to the desired row ID to read in the dat file.
#########################################################################
proc test {libref} {
    global dbc_libref
    global dbc_retrieve_var_list
    global TEST
    set TEST "1"
    set dbc_libref $libref
    set dbc_retrieve_var_list "libref Name Category Type Ref_Temp"

        puts "#########################################################################"
        puts "    Init DB"
        DBC_init_db

        puts "#########################################################################"
        puts "    call retrieve"
        DBC_retrieve
}

#########################################################################
##  Added this proc to override the default which writes out to the
##  dbc_information.out file.   If you want the dbc_information.out file
##  created comment out this proc, The existance of this file may cause
##  when running under UGmanager.
#########################################################################
proc MOM__util_print { line } { }
