###############################################################################
# mvision.tcl - DBC Event Handler for material DB stored in E-Mvision
###############################################################################
#
# The following fields may need to be customized by a customer to run
# the local eMvision installation:
#
# dbc_databank        "GMMPD_UGDEMO"      The eMision databank
# dbc_emv_login       "guest"             eMvision login ID
# dbc_emv_password    "emv"               eMvision password
# dbc_group           "All_Materials"     eMvision group name
# dbc_libref_id       "RE_SER_NUM"        column heading for the material libref
# dbc_name_id         "A2_MATERIAL_SPEC"  column heading for the material name
# dbc_category_id     "A1_CATEGORY"       column heading for the material category
# dbc_temp_id         "A4_TEMP"           column heading for the material reference temperature
# dbc_units           "US-Consistent"     eMvision output units
# dbc_debug_props     " "                 A full OS filename for debug output of a material
# dbc_debug_list      " "                 Debug output of all materials in databank
#
# There are three material record types in the material library. The first
# five fields are the same in each:
#
#         Libref Name Category Type Ref_Temp
#
# The Version field has been added since. This is to be found as the last field 
#         of each record.
#
# When Type == ISO (ISOTROPIC)
#
#         Youngs_Modulus Poissons_Ratio Shear_Modulus Mass_Density 
#         Thermal_Exp_Coeff Thermal_Cond Plastic_Strain_Ratio
#         Work_Hardening Forming_Limit Stress_Strain 
#         Yield Ultimate_Tensile
#         Fatigue_Strength_Coef Fatigue_Strength_Exp
#         Fatigue_Ductility_Coef Fatigue_Ductility_Exp
#
# When Type == ORTHO (ORTHOTROPIC)
#
#         Youngs_Modulus_X Youngs_Modulus_Y Youngs_Modulus_Z 
#         Poissons_Ratio_X Poissons_Ratio_Y Poissons_Ratio_Z 
#         Shear_Modulus_X Shear_Modulus_Y Shear_Modulus_Z Mass_Density  
#         Thermal_Exp_Coeff_X Thermal_Exp_Coeff_Y Thermal_Exp_Coeff_Z 
#         Thermal_Cond_X Thermal_Cond_Y Thermal_Cond_Z 
#
# When Type == ANISO (ANISOTROPIC)
#
#         G_1_1
#         G_2_1 G_2_2  
#         G_3_1 G_3_2 G_3_3  
#         G_4_1 G_4_2 G_4_3 G_4_4  
#         G_5_1 G_5_2 G_5_3 G_5_4 G_5_5 
#         G_6_1 G_6_2 G_6_3 G_6_4 G_6_5 G_6_6
#         Mass_Density 
#         Thermal_Exp_Coeff_1 Thermal_Exp_Coeff_2 Thermal_Exp_Coeff_3
#         Thermal_Exp_Coeff_4 Thermal_Exp_Coeff_5 Thermal_Exp_Coeff_6 
#         Thermal_Cond_1_1 
#         Thermal_Cond_2_1 Thermal_Cond_2_2 
#         Thermal_Cond_3_1 Thermal_Cond_3_2 Thermal_Cond_3_3 
#         Specific_Heat
##
#         Libref   is a number string (>0) which is an unique identifier in the
#                  library
#         Name     is a string
#         Category is METAL or
#                     PLASTIC or
#                     OTHER
#         Type     is ISO   (ISOTROPIC) or
#                     ORTHO (ORTHOTROPIC) or 
#                     ANISO (ANISOTROPIC)
#         Version is a string. 
#         Ref_Temp is a double
#         Plastic_Strain_Ratio is a double
#         All others are doubles or Temperature Table references.
#
# Temperature Table references may contain either a single scalar
# value, or may contain the name of a temperature table record.
# The temperature table records must begin with the letter 't' or 'T'. 
# Each field is delimited by a blank space.
# The second token of the temperature table is the count of the number of 
# temperature-value pairs. The remaining tokens are the temperature-value
# pairs.
#
# Table format example:
#
# Table 3 0.0 23.5 50 1.456e2 100. 2.34e3
# The above table has 3 temperature-value pairs as follows:
#     0.0  - 23.5
#     50   - 1.456e2
#     100. - 2.34e3
#
# If a field of a record is to be not set, code the string "EMPTY" to occupy
# the position of that field.
#
# Queries against string fields (only name at this time) are by default
# case insensitive. This is done in DBC_create_criterion by folding the
# criteria string to upper case and then folding the dataBase field to
# upper case in DBC_execute_query as the strings are compared.
# To escape away from case insensitive queries, that is, to query against
# a specific case of a character, enclose the criteria string in {}.
#
# 
#
###############################################################################
#MOM_set_debug_mode on
#---------------------------------------------
# Non DBC Global Variables
#---------------------------------------------

# set UGII_USER_MATERIAL_LIB_MODULE to $UGII_ROOT_DIR\\ematl.dll.
set dbExecName "[MOM_ask_env_var UGII_USER_MATERIAL_LIB_MODULE]"

set lastRowNumber  0
set DEBUG "0"
#set debug_tcl       [open "C:\\debug.txt" w ]

#---------------------------------------------
# Global variables set by DBC for Input/Output
#---------------------------------------------
set dbc_lhs_exp     ""
set dbc_rhs_exp     ""
set dbc_relop       ""
set dbc_query       ""
set dbc_subqry1     ""
set dbc_subqry2     ""
set dbc_boolop      ""
set dbc_class_name  ""
set dbc_attr_count  0
set dbc_attr_id     ""
set dbc_query_count 0
set dbc_libref      ""
set dbc_var_list    ""

set dbc_num_attributes   70
set dbc_databank        "GMMPD_UGDEMO"
set dbc_emv_api         "[MOM_ask_env_var EMV_API_PROPERTIES_FILE]"
set dbc_emv_dir         "[MOM_ask_env_var EMV_DIR]"
set dbc_emv_login       "snyderk"
set dbc_emv_password    "kens"
set dbc_emv_library     "MscEmvApi"
set dbc_group           "All_Materials"
set dbc_libref_id       "RE_SER_NUM"
set dbc_name_id         "A2_MATERIAL_SPEC"
set dbc_category_id     "A1_CATEGORY"
set dbc_temp_id         "A4_TEMP"
set dbc_units           "SI_Standard_Consistent_m_kg_s_K_Pa"
set dbc_debug_props     " "
set dbc_debug_list      " "
#set dbc_debug_props     "C:\\props.xml"
#set dbc_debug_list      "C:\\list.txt"

#---------------------------------------------
proc DBC_init_db {} {
#---------------------------------------------
   global DEBUG
   global dataBase
   global lastRowNumber
   global dbFileName
   global dbExecName
   global line
   global dbc_num_attributes
   global dbc_data_reference
   global dbc_row
   global dbc_librefs
   global dbc_names
   global dbc_categories
   global dbc_types
   global dbc_ref_temps

   set dbc_data_reference "EMPTY"
   set row 0
   
   # dbc_row will use a return code as well as Row number 
   # If dbc_row is negative EMATL_main failed and dbc_row is the negative 
   # of the error code 
   set dbc_row -1
   MOM_run_user_function $dbExecName EMATL_main

   if {$dbc_row < 0 }  \
      { 
         set message "Error executing EMATL command from $dbExecName."
         set message \
          "$message \n Return Code = $dbc_row"
         return -code error $message
      }
   # Read the Mvision database file into a 2-dimensional array 'dataBase'
    for {set db_row 0} {$db_row < $dbc_row} {incr db_row 1} {
     
      set first_token [string range $dbc_librefs($db_row) 0 0]
      
      if {"$first_token" != "t"       \
       && "$first_token" != "T"       \
       && "$first_token" != "#"}      {
          # Read the first 5 tokens. 
          # These are common to all materials.
          set dataBase($db_row,0) $dbc_librefs($db_row)
          set dataBase($db_row,1) $dbc_names($db_row)
          set dataBase($db_row,2) $dbc_categories($db_row)
          set dataBase($db_row,3) $dbc_types($db_row)
          set dataBase($db_row,4) $dbc_ref_temps($db_row)

# DLG Setting ref temp to empty, the field in DB is not the what UG is looking for.
          set dataBase($db_row,4) "EMPTY"
          }  
          # Read the first 5 tokens.

       # Set the rest of the array to blanks.
       # This is needed only because of a bug where all the
       # variables need to be in the rset.
       set dataBase($db_row,5) " "
       set dataBase($db_row,6) " "
       set dataBase($db_row,7) " "
       set dataBase($db_row,8) " "
       set dataBase($db_row,9) " "
       set dataBase($db_row,10) " "
       set dataBase($db_row,11) " "
       set dataBase($db_row,12) " "
       set dataBase($db_row,13) " "
       set dataBase($db_row,14) " "
       set dataBase($db_row,15) " "
       set dataBase($db_row,16) " "
       set dataBase($db_row,17) " "
       set dataBase($db_row,18) " "
       set dataBase($db_row,19) " "
       set dataBase($db_row,20) " "
       set dataBase($db_row,21) " "
       set dataBase($db_row,22) " "
       set dataBase($db_row,23) " "
       set dataBase($db_row,24) " "
       set dataBase($db_row,25) " "
       set dataBase($db_row,26) " "
       set dataBase($db_row,27) " "
       set row [expr $row + 1] 
      
   }
   set lastRowNumber [expr $row ]

   if { "$DEBUG" == "1" } {
      puts "The database is from file $dbFileName"
      puts "libref	Name	Category	Type		Ref_Temp"

      for {set ix 0} {$ix <= $lastRowNumber} {set ix [expr $ix + 1]} {
         puts "$dataBase($ix,0)	$dataBase($ix,1)	$dataBase($ix,2)	$dataBase($ix,3)	$dataBase($ix,4)"
      }
   }
}

#---------------------------------------------
proc DBC_translate_att_alias {} {
#---------------------------------------------
   global DEBUG
   global dbc_alias
   global dbc_id

   if { "$DEBUG" == "1" } {
      puts "DBC_translate_att_alias(dbc_alias = $dbc_alias, dbc_id = $dbc_id)" 
   }

   # The input dbc_id is the column in the RSET
   # The variable db_row is a local variable in execute_query

   set dbc_id "\$dataBase(\$db_row,$dbc_id)"

   if { "$DEBUG" == "1" } {
      puts "DBC_Translate_Att_Alias made: $dbc_id"
   }

}

#---------------------------------------------
proc DBC_create_criterion {} {
#---------------------------------------------
   global DEBUG
   global dbc_lhs_exp
   global dbc_rhs_exp
   global dbc_relop
   global dbc_query
   global fpout


   if { "$DEBUG" == "1" } {
      puts "DBC_create_criterion(dbc_lhs_exp = $dbc_lhs_exp, dbc_rhs_exp = $dbc_rhs_exp, dbc_relop = $dbc_relop)"
   }

   # For now just assume lhs is of form db(alias)
   scan $dbc_lhs_exp "DB(%s" x
   
   if { "$DEBUG" == "1" } {
      puts "x = $x"
   }
   set y [string last ) $x]
   set y [expr $y - 1]
   set dbc_lhs_exp [string range $x 0 $y]

   set dbc_query " $dbc_lhs_exp $dbc_relop \"$dbc_rhs_exp\" "

   if {$dbc_relop == "=="}  {
       set dbc_command_prefix "\[ string match "

       # Fold the string to upper if the first char is NOT "LEFT-BRACE"
       # Cannot type the actual char here because it gets parsed
       set dcb_initial_char [string index $dbc_rhs_exp 0]
       set dcb_toupper " "
       set dcb_closer " "
 
       if { $dcb_initial_char != "\{" }  {
           set dbc_rhs_exp [string toupper $dbc_rhs_exp]
           set dcb_toupper " \[ string toupper"
           set dcb_closer " \]"
       }

       set dbc_query "$dbc_command_prefix $dbc_rhs_exp $dcb_toupper $dbc_lhs_exp $dcb_closer \]"
   }

   if { "$DEBUG" == "1" } {
      puts "DBC_Create_Criterion made: $dbc_query"
   }
}

#---------------------------------------------
proc DBC_create_query {} {
#---------------------------------------------
   global DEBUG
   global dbc_query
   global dbc_subqry1
   global dbc_subqry2
   global dbc_boolop

   if { "$DEBUG" == "1" } {
      puts "DBC_create_query(dbc_subqry1 = $dbc_subqry1, dbc_subqry2 = $dbc_subqry2, dbc_boolop = $dbc_boolop)"
   }

   set dbc_query "$dbc_subqry1 $dbc_boolop $dbc_subqry2"

   if { "$DEBUG" == "1" } {
      puts "DBC_Create_Query made: $dbc_query"
      
   set $dbc_subqry1 " "
   set $dbc_subqry2 " "

   }
}

#---------------------------------------------
proc DBC_execute_query {} {
#---------------------------------------------
   global DEBUG
   global dbc_class_name
   global dbc_query
   global dbc_attr_count
   global dbc_attr_aliases
   global dbc_attr_id
   global lastRowNumber
   global dataBase

   if { "$DEBUG" == "1" } {
      puts "DBC_execute_query(dbc_class_name = $dbc_class_name, dbc_query = $dbc_query)"
   }

   DBC_load_rset_data # Load the structure of the RSET

   set rset_row 0

   for {set db_row 0} {$db_row < $lastRowNumber} {incr db_row 1} {
      if { [expr $dbc_query]  } {
         for {set ix 0} {$ix < $dbc_attr_count} { incr ix 1 } {
            DBC_ask_attr_id $dbc_attr_aliases($ix)
            DBC_write_to_rset $rset_row $dbc_attr_aliases($ix) \
                                $dataBase($db_row,$dbc_attr_id)
         }
         incr rset_row 1
      }
   }

   if { "$DEBUG" == "1" } {
      puts "DBC_Execute_Query: Query $dbc_query, RSET rows: $rset_row"
   }
}

#---------------------------------------------
proc DBC_execute_query_for_count {} {
#---------------------------------------------
   global DEBUG
   global dbc_class_name
   global dbc_query
   global dbc_query_count
   global lastRowNumber
   global dataBase

   if { "$DEBUG" == "1" } {
      puts "DBC_execute_query_for_count(dbc_class_name = $dbc_class_name, dbc_query = $dbc_query)"
   }

   set dbc_query_count 0

   for {set db_row 0} {$db_row < $lastRowNumber} {incr db_row 1} {
      if { [expr $dbc_query]  } {
         incr dbc_query_count 1
      }
   }

   if { "$DEBUG" == "1" } {
      puts "DBC_Execute_Query_For_Count: Query $dbc_query, Count: $dbc_query_count"
   }
   
   set dbc_query " "
}


#---------------------------------------------
proc DBC_retrieve {} {
#---------------------------------------------
   global DEBUG
   global debug_tcl

   global dbc_libref
   global dbc_retrieve_var_list
   global lastRowNumber
   global dbc_data_reference 
   global dataBase
   global dbFileName
   global dbExecName

   # These variables were documented by the matl_lib application.
   # The matl_lib UG Object will be asking for these values.
   
   # Common to all materials.
   global dbc_Name
   global dbc_Category
   global dbc_Type
   global dbc_Ref_Temp
   global dbc_Version
   
   # Isotropic Materials
   global dbc_Youngs_Modulus
   global dbc_Poissons_Ratio
   global dbc_Shear_Modulus
   global dbc_Mass_Density
   global dbc_Thermal_Exp_Coeff
   global dbc_Thermal_Cond
   global dbc_Plastic_Strain_Ratio
   global dbc_Work_Hardening
   global dbc_Forming_Limit
   global dbc_Stress_Strain
   global dbc_Yield
   global dbc_Ultimate_Tensile
   global dbc_Specific_Heat
   global dbc_Fatigue_Strength_Coef
   global dbc_Fatigue_Strength_Exp
   global dbc_Fatigue_Ductility_Coef
   global dbc_Fatigue_Ductility_Exp
      
   # Orthotropic Materials
   global dbc_Youngs_Modulus_X
   global dbc_Youngs_Modulus_Y
   global dbc_Youngs_Modulus_Z
   global dbc_Poissons_ratio_X
   global dbc_Poissons_ratio_Y
   global dbc_Poissons_ratio_Z
   global dbc_Shear_modulus_X
   global dbc_Shear_modulus_Y
   global dbc_Shear_modulus_Z
   global dbc_Thermal_Exp_Coeff_X
   global dbc_Thermal_Exp_Coeff_Y
   global dbc_Thermal_Exp_Coeff_Z
   global dbc_Thermal_Cond_X
   global dbc_Thermal_Cond_Y
   global dbc_Thermal_Cond_Z
   
   # Anisotropic Materials
   global dbc_G_1_1
   global dbc_G_2_1
   global dbc_G_2_2
   global dbc_G_3_1
   global dbc_G_3_2
   global dbc_G_3_3
   global dbc_G_4_1
   global dbc_G_4_2
   global dbc_G_4_3
   global dbc_G_4_4
   global dbc_G_5_1
   global dbc_G_5_2
   global dbc_G_5_3
   global dbc_G_5_4
   global dbc_G_5_5
   global dbc_G_6_1
   global dbc_G_6_2
   global dbc_G_6_3
   global dbc_G_6_4
   global dbc_G_6_5
   global dbc_G_6_6
   global dbc_Thermal_Exp_Coeff_1
   global dbc_Thermal_Exp_Coeff_2
   global dbc_Thermal_Exp_Coeff_3
   global dbc_Thermal_Exp_Coeff_4
   global dbc_Thermal_Exp_Coeff_5
   global dbc_Thermal_Exp_Coeff_6
   global dbc_Thermal_Cond_1_1
   global dbc_Thermal_Cond_2_1
   global dbc_Thermal_Cond_2_2
   global dbc_Thermal_Cond_3_1
   global dbc_Thermal_Cond_3_2
   global dbc_Thermal_Cond_3_3
   
   global table_string
   global whole_line

   if { "$DEBUG" == "1" }  \
   {
      puts "DBC_retrieve(libref = $dbc_libref)"
      puts "DBC_retrieve(lastRowNumber = $lastRowNumber)"
   }

   set dbc_data_reference $dbc_libref 

   for {set db_row 0} {$db_row < $lastRowNumber} {incr db_row 1} {

      # Search for the correct record.
      # For each row search, step through the file at the same time.
      
      if { $dbc_libref == $dataBase($db_row,0)}   {  
         #  Find the row for the desired LibRef

         if { "$DEBUG" == "1" }  {
            puts "Found libref in row $db_row"
            puts "$dbc_Name"
            puts "$dbc_Category"
            puts "Material Type is $dbc_Type"
            }

         break  
         # Stop the search loop.
         }  
         #  Find the row for the desired LibRef
      } 
      # Search for the correct record.
       
   set dbc_Name     $dataBase($db_row,1)
   set dbc_Category $dataBase($db_row,2)
   set dbc_Type     $dataBase($db_row,3)
   set dbc_Ref_Temp $dataBase($db_row,4)

   # EMATL_main sets the following variables:
   #   dbc_libref
   #   dbc_Youngs_Modulus
   #   dbc_Poissons_Ratio
   #   dbc_Shear_Modulus
   #   dbc_Mass_Density
   #   dbc_Thermal_Exp_Coeff
   #   dbc_Thermal_Cond
   #   dbc_Plastic_Strain_Ratio
   #   dbc_Work_Hardening
   #   dbc_Forming_Limit
   #   dbc_Stress_Strain
   #   dbc_Yield
   #   dbc_Ultimate_Tensile
   #   dbc_Specific_Heat
   #   dbc_Fatigue_Strength_Coef
   #   dbc_Fatigue_Strength_Exp
   #   dbc_Fatigue_Ductility_Coef
   #   dbc_Fatigue_Ductility_Exp
   MOM_run_user_function $dbExecName EMATL_main

# DLG Setting ref temp to empty, the field in DB is not the what UG is looking for.
   set dbc_Ref_Temp "EMPTY" 

  # Set the variables based upon the values of this row 
  # (for the desired LibRef)
  switch $dbc_Type  {   
      ISO  { 
      
           # Convert the "SI_Standard_Consistent_m_kg_s_K_Pa" units to 
           # UGS consistant metric units
           # The conversion is done here to allow user customization.
           # WARNING:: All values not converted are assumed to be correct.
           # WARNING:: but this has not been verified because the Mvision data
           # WARNING:: we have seen was very limited.
            
           set table_string $dbc_Mass_Density
           DBC_convert_temperature_table [expr 1./1000000000.]
           set dbc_Mass_Density $table_string
           
           set table_string $dbc_Youngs_Modulus
           DBC_convert_temperature_table [expr 1./1000.]
           set dbc_Youngs_Modulus $table_string
                                
           set table_string $dbc_Shear_Modulus
           DBC_convert_temperature_table [expr 1./1000.]
           set dbc_Shear_Modulus $table_string
                
           set table_string $dbc_Thermal_Exp_Coeff
           DBC_convert_temperature_table [expr 1./1000.]
           set dbc_Thermal_Exp_Coeff $table_string
                
           set table_string $dbc_Thermal_Cond
           DBC_convert_temperature_table [expr 1./1000.]
           set dbc_Thermal_Cond $table_string
                
           set table_string $dbc_Yield
           DBC_convert_temperature_table [expr 1./1000.]
           set dbc_Yield $table_string
                
           set table_string $dbc_Ultimate_Tensile
           DBC_convert_temperature_table [expr 1./1000.]
           set dbc_Ultimate_Tensile $table_string
                                
                               
           if { "$DEBUG" == "2" }  {
                  puts $debug_tcl "$dbc_libref"
                  puts $debug_tcl "$dbc_Ref_Temp"
                  puts $debug_tcl "$dbc_Youngs_Modulus"
                  puts $debug_tcl "$dbc_Poissons_Ratio"
                  puts $debug_tcl "$dbc_Shear_Modulus"
                  puts $debug_tcl "$dbc_Mass_Density"
                  puts $debug_tcl "$dbc_Thermal_Exp_Coeff"
                  puts $debug_tcl "$dbc_Thermal_Cond"
                  puts $debug_tcl "$dbc_Plastic_Strain_Ratio"
                  puts $debug_tcl "$dbc_Yield"
                  puts $debug_tcl "$dbc_Ultimate_Tensile"
                  puts $debug_tcl "$dbc_Work_Hardening"
                  puts $debug_tcl "$dbc_Forming_Limit"
                  puts $debug_tcl "$dbc_Stress_Strain"
                  puts "$dbc_Specific_Heat"
                  puts "$dbc_Fatigue_Strength_Coef"
                  puts "$dbc_Fatigue_Strength_Exp"
                  puts "$dbc_Fatigue_Ducility_Coef"
                  puts "$dbc_Fatigue_Ductility_Exp"
                  puts "$dbc_Version"
               }

           }
           # Material type ISO
     ORTHO {
           
           if { "$DEBUG" == "1" }  {
                  puts "$dbc_Ref_Temp"
                  puts "$dbc_Youngs_Modulus_X"
                  puts "$dbc_Youngs_Modulus_Y"
                  puts "$dbc_Youngs_Modulus_Z"
                  puts "$dbc_Poissons_ratio_X"
                  puts "$dbc_Poissons_ratio_Y"
                  puts "$dbc_Poissons_ratio_Z"
                  puts "$dbc_Shear_modulus_X"
                  puts "$dbc_Shear_modulus_Y"
                  puts "$dbc_Shear_modulus_Z"
                  puts "$dbc_Mass_Density"
                  puts "$dbc_Thermal_Exp_Coeff_X"
                  puts "$dbc_Thermal_Exp_Coeff_Y"
                  puts "$dbc_Thermal_Exp_Coeff_Z"
                  puts "$dbc_Thermal_Cond_X"
                  puts "$dbc_Thermal_Cond_Y"
                  puts "$dbc_Thermal_Cond_Z"
                  puts "$dbc_Version"
               }

           }
           # Material type ORTHO
           
     ANISO { 

           if { "$DEBUG" == "1" }  {
                  puts "$dbc_Ref_Temp"
                  puts "$dbc_G_1_1"
                  puts "$dbc_G_2_1"
                  puts "$dbc_G_2_2"
                  puts "$dbc_G_3_1"
                  puts "$dbc_G_3_2"
                  puts "$dbc_G_3_3"
                  puts "$dbc_G_4_1"
                  puts "$dbc_G_4_2"
                  puts "$dbc_G_4_3"
                  puts "$dbc_G_4_4"
                  puts "$dbc_G_5_1"
                  puts "$dbc_G_5_2"
                  puts "$dbc_G_5_3"
                  puts "$dbc_G_5_4"
                  puts "$dbc_G_5_5"
                  puts "$dbc_G_6_1"
                  puts "$dbc_G_6_2"
                  puts "$dbc_G_6_3"
                  puts "$dbc_G_6_4"
                  puts "$dbc_G_6_5"
                  puts "$dbc_G_6_6"
                  puts "$dbc_Mass_Density"
                  puts "$dbc_Thermal_Exp_Coeff_1"
                  puts "$dbc_Thermal_Exp_Coeff_2"
                  puts "$dbc_Thermal_Exp_Coeff_3"
                  puts "$dbc_Thermal_Exp_Coeff_4"
                  puts "$dbc_Thermal_Exp_Coeff_5"
                  puts "$dbc_Thermal_Exp_Coeff_6"
                  puts "$dbc_Thermal_Cond_1_1"
                  puts "$dbc_Thermal_Cond_2_1"
                  puts "$dbc_Thermal_Cond_2_2"
                  puts "$dbc_Thermal_Cond_3_1"
                  puts "$dbc_Thermal_Cond_3_2"
                  puts "$dbc_Thermal_Cond_3_3"
                  puts "$dbc_Version"
               }
           }
           # Material type ANISO
         }   
         # for each material type.
      
}

#---------------------------------------------
proc DBC_convert_temperature_table { conversion_factor } {
#---------------------------------------------
    global table_string
    global first_token
    global whole_line

    set new_table_string $table_string
    
    set whole_line $table_string

    DBC_scan_and_strip
    if {"$first_token" != "EMPTY" } {
    
        set new_table_string $first_token

        set first_token [string range $first_token 0 0]

        if {"$first_token" == "T" || "$first_token" == "t"}   {
            
            # Get the point count
            DBC_scan_and_strip
            set point_number $first_token
            set new_table_string  "$new_table_string $first_token"
            
            for {set point_cnt 0} {$point_cnt < $point_number} {incr point_cnt 1} {
                
                # X-value
                DBC_scan_and_strip  
                set new_table_string  "$new_table_string $first_token"
        
                # Y-value
                DBC_scan_and_strip  
                set first_token [expr $first_token*$conversion_factor]
                set new_table_string  "$new_table_string $first_token"
            
            }  
            #  for each point_cnt
        
        }  else {
                # Convert the single scalar
                set new_table_string [expr $table_string*$conversion_factor]
                } 
    }
    
    set table_string $new_table_string
    
}

#---------------------------------------------
proc DBC_scan_and_strip {} {
#
#  Because this Tcl implementation supports only 20 tokens in scan,
#  This function will scan for the next token,
#  measure its length, then strip the token from the record.
#
#---------------------------------------------
   global DEBUG
   global first_token
   global whole_line

   scan $whole_line "%s" first_token
   
   set token_len [string length $first_token]

   set whole_line [string trimleft $whole_line]
   set whole_line [string range $whole_line [expr $token_len + 1] end]
      
   if { "$DEBUG" == "1" } {
       puts "DBC_scan_and_strip: first_token = $first_token"
       puts "DBC_scan_and_strip: token_len = $token_len"
       puts "DBC_scan_and_strip: whole_line = $whole_line"
                          }

}
#########################################################################
##  Added this proc to override the default which writes out to the 
##  dbc_information.out file.   If you want the dbc_information.out file
##  created comment out this proc, The existance of this file may cause 
##  when running under UGmanager.
#########################################################################
proc MOM__util_print { line } { }
